=begin
 Copyright (C) 2000, 2001, 2002 RiskMap srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
=end

# $Id: montecarlo_pricers.rb,v 1.5 2002/01/16 15:17:06 nando Exp $

require 'QuantLib'
require 'runit/testcase'
require 'runit/testsuite'
require 'runit/cui/testrunner'


class MonteCarloPricerTest < RUNIT::TestCase
    def name
        "Testing Monte Carlo pricers..."
    end
    def test
        seed         = 3456789
        fixedSamples = 100
        minimumTol   = 0.01

        # data from "Implementing Derivatives Model", Clewlow, Strickland, pag.118-123
        cases = [
            [QuantLib::DiscreteGeometricAPO,
              "Call", 100.0, 100.0, 0.03, 0.06,
              [0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0],
              0.2, 5.34255485619]
        ]

        cases.each {
            |pricer,type,underlying,strike,qRate,rRate,times,vol,expected|

            value = pricer.new(type, underlying, strike,
                               qRate, rRate, times, vol).value

            unless (value-expected).abs <= 1.0e-10
                assert_fail(<<-MESSAGE

    In the first batch of cases:
        calculated value: #{value}
        stored value:     #{expected}

                    MESSAGE
                )
            end
        }


        # data from "Option Pricing Formulas", Haug, pag.96-97
        cases = [
            [QuantLib::EuropeanOption,         "Put", 80.0, 85.0, -0.03, 0.05, 0.25, 0.2, 5.21858890396],
            [QuantLib::ContinuousGeometricAPO, "Put", 80.0, 85.0, -0.03, 0.05, 0.25, 0.2, 4.69221973405]
        ]

        cases.each {
            |pricer,type,underlying,strike,qRate,rRate,resTime,vol,expected|

            value = pricer.new(type, underlying, strike,
                               qRate, rRate, resTime, vol).value()

            unless (value-expected).abs <= 1.0e-10
                assert_fail(<<-MESSAGE

    In the second batch of cases:
        calculated value: #{value}
        stored value:     #{expected}

                    MESSAGE
                )
            end
        }


        # trying to approximate the continuous version with the discrete version
        cases = [
            [QuantLib::DiscreteGeometricAPO, "Put", 80.0, 85.0, -0.03, 0.05, 0.25, 90000, 0.2, 4.6922231469]
        ]

        cases.each {
            |pricer,type,underlying,strike,qRate,rRate,resTime,tSteps,vol,expected|

            dt = resTime/tSteps
            times = (0...tSteps).map { |i| (i+1)*dt }
            value = pricer.new(type, underlying, strike,
                               qRate, rRate, times, vol).value()

            unless (value-expected).abs <= 1.0e-10
                assert_fail(<<-MESSAGE

    In the third batch of cases:
        calculated value: #{value}
        stored value:     #{expected}

                    MESSAGE
                )
            end
        }


        cases = [
            [QuantLib::McEuropean,      "Put", 80.0, 85.0, -0.03, 0.05, 0.25, 0.2, 5.91358723580, false],
            [QuantLib::McEuropean,      "Put", 80.0, 85.0, -0.03, 0.05, 0.25, 0.2, 5.42005964479,  true],
            [QuantLib::McEuropean,     "Call", 80.0, 85.0, -0.03, 0.05, 0.25, 0.2, 1.98816310759, false],
            [QuantLib::McEuropean,     "Call", 80.0, 85.0, -0.03, 0.05, 0.25, 0.2, 2.12098432917,  true],
            [QuantLib::McEuropean, "Straddle", 80.0, 85.0, -0.03, 0.05, 0.25, 0.2, 7.90175034339, false],
            [QuantLib::McEuropean, "Straddle", 80.0, 85.0, -0.03, 0.05, 0.25, 0.2, 7.54104397396,  true]
        ]

        cases.each {
            |pricer,type,underlying,strike,qRate,rRate,resTime,vol,expected,isAntithetic|

            p = pricer.new(type, underlying, strike,
                           qRate, rRate, resTime, vol,
                           isAntithetic,seed)
            value = p.valueWithSamples(fixedSamples)

            unless (value-expected).abs <= 1.0e-10
                assert_fail(<<-MESSAGE

    In the fourth batch of cases:
        calculated value: #{value}
        stored value:     #{expected}

                    MESSAGE
                )
            end

            tol = p.errorEstimate/value
            tol = [tol/2.0,minimumTol].min
            value = p.value(tol)
            accuracy = p.errorEstimate/value
            unless accuracy <= tol
                assert_fail(<<-MESSAGE

    In the fourth batch of cases:
        achieved accuracy: #{accuracy}
        requested:         #{tol}

                    MESSAGE
                )
            end

        }


        # data from "Asian Option", Levy, 1997
        # in "Exotic Options: The State of the Art", edited by Clewlow, Strickland
        cases = [
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                      0.0, 11.0/12.0,    2, 0.13, 1.38418414762, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                      0.0, 11.0/12.0,    4, 0.13, 1.57691714387, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                      0.0, 11.0/12.0,    8, 0.13, 1.66062743445, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                      0.0, 11.0/12.0,   12, 0.13, 1.68847081883, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                      0.0, 11.0/12.0,   26, 0.13, 1.72955964448, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                      0.0, 11.0/12.0,   52, 0.13, 1.73372169316, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                      0.0, 11.0/12.0,  100, 0.13, 1.74918801089, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                      0.0, 11.0/12.0,  250, 0.13, 1.75421310915, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                      0.0, 11.0/12.0,  500, 0.13, 1.75158383443, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                      0.0, 11.0/12.0, 1000, 0.13, 1.75162110180, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                 1.0/12.0, 11.0/12.0,    2, 0.13, 1.83665087164, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                 1.0/12.0, 11.0/12.0,    4, 0.13, 2.00560271429, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                 1.0/12.0, 11.0/12.0,    8, 0.13, 2.07789721712, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                 1.0/12.0, 11.0/12.0,   12, 0.13, 2.09622556625, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                 1.0/12.0, 11.0/12.0,   26, 0.13, 2.14229795212, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                 1.0/12.0, 11.0/12.0,   52, 0.13, 2.14470270916, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                 1.0/12.0, 11.0/12.0,  100, 0.13, 2.15954145741, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                 1.0/12.0, 11.0/12.0,  250, 0.13, 2.16007690017, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                 1.0/12.0, 11.0/12.0,  500, 0.13, 2.15986704400, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                 1.0/12.0, 11.0/12.0, 1000, 0.13, 2.15951634387, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                 3.0/12.0, 11.0/12.0,    2, 0.13, 2.63315092584, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                 3.0/12.0, 11.0/12.0,    4, 0.13, 2.76723962361, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                 3.0/12.0, 11.0/12.0,    8, 0.13, 2.83124836881, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                 3.0/12.0, 11.0/12.0,   12, 0.13, 2.84290301412, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                 3.0/12.0, 11.0/12.0,   26, 0.13, 2.88179560417, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                 3.0/12.0, 11.0/12.0,   52, 0.13, 2.88447044543, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                 3.0/12.0, 11.0/12.0,  100, 0.13, 2.89985329603, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                 3.0/12.0, 11.0/12.0,  250, 0.13, 2.90047296063, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                 3.0/12.0, 11.0/12.0,  500, 0.13, 2.89813412160, true, true],
            [QuantLib::McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,
                 3.0/12.0, 11.0/12.0, 1000, 0.13, 2.89703362437, true, true]
        ]

        cases.each {
            |pricer,type,underlying,strike,qRate,rRate,first,length,fixings,vol,
             expected,isAntithetic,isControlVariated|

            dt=(length)/(fixings-1)
            times = (0...fixings).map { |i| first + i*dt }

            p = pricer.new(type, underlying, strike,
                               qRate, rRate, times, vol,
                               isAntithetic, isControlVariated,seed)

            value = p.valueWithSamples(fixedSamples)
            unless (value-expected).abs <= 1.0e-10
                assert_fail(<<-MESSAGE

    In the fifth batch of cases:
        calculated value: #{value}
        stored value:     #{expected}

                    MESSAGE
                )
            end

            tol = p.errorEstimate/value
            tol = [tol/2.0,minimumTol].min
            value = p.value(tol)
            accuracy = p.errorEstimate/value
            unless accuracy <= tol
                assert_fail(<<-MESSAGE

    In the fifth batch of cases:
        achieved accuracy: #{accuracy}
        requested:         #{tol}

                    MESSAGE
                )
            end
        }


        # data from "Asian Option", Levy, 1997
        # in "Exotic Options: The State of the Art", edited by Clewlow, Strickland
        cases = [
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                      0.0, 11.0/12.0,    2, 0.13, 1.51917595129, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                      0.0, 11.0/12.0,    4, 0.13, 1.67940165674, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                      0.0, 11.0/12.0,    8, 0.13, 1.75371215251 , true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                      0.0, 11.0/12.0,   12, 0.13, 1.77595318693, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                      0.0, 11.0/12.0,   26, 0.13, 1.81430536630, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                      0.0, 11.0/12.0,   52, 0.13, 1.82269246898, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                      0.0, 11.0/12.0,  100, 0.13, 1.83822402464, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                      0.0, 11.0/12.0,  250, 0.13, 1.83875059026, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                      0.0, 11.0/12.0,  500, 0.13, 1.83750703638 , true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                      0.0, 11.0/12.0, 1000, 0.13, 1.83887181884 , true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                 1.0/12.0, 11.0/12.0,    2, 0.13, 1.51154400089, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                 1.0/12.0, 11.0/12.0,    4, 0.13, 1.67103508506, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                 1.0/12.0, 11.0/12.0,    8, 0.13, 1.74529684070, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                 1.0/12.0, 11.0/12.0,   12, 0.13, 1.76667074564, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                 1.0/12.0, 11.0/12.0,   26, 0.13, 1.80528400613, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                 1.0/12.0, 11.0/12.0,   52, 0.13, 1.81400883891, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                 1.0/12.0, 11.0/12.0,  100, 0.13, 1.82922901451, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                 1.0/12.0, 11.0/12.0,  250, 0.13, 1.82937111773, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                 1.0/12.0, 11.0/12.0,  500, 0.13, 1.82826193186, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                 1.0/12.0, 11.0/12.0, 1000, 0.13, 1.82967846654, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                 3.0/12.0, 11.0/12.0,    2, 0.13, 1.49648170891, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                 3.0/12.0, 11.0/12.0,    4, 0.13, 1.65443100462, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                 3.0/12.0, 11.0/12.0,    8, 0.13, 1.72817806731, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                 3.0/12.0, 11.0/12.0,   12, 0.13, 1.74877367895, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                 3.0/12.0, 11.0/12.0,   26, 0.13, 1.78733801988, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                 3.0/12.0, 11.0/12.0,   52, 0.13, 1.79624826757, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                 3.0/12.0, 11.0/12.0,  100, 0.13, 1.81114186876, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                 3.0/12.0, 11.0/12.0,  250, 0.13, 1.81101152587, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                 3.0/12.0, 11.0/12.0,  500, 0.13, 1.81002311939, true, true],
            [QuantLib::McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,
                 3.0/12.0, 11.0/12.0, 1000, 0.13, 1.81145760308, true, true]
        ]

        cases.each {
            |pricer,type,underlying,strike,qRate,rRate,first,length,fixings,vol,
             expected,isAntithetic,isControlVariated|

            dt=(length)/(fixings-1)
            times = (0...fixings).map { |i| first + i*dt }

            p = pricer.new(type, underlying, qRate, rRate, times,
                               vol, isAntithetic, isControlVariated, seed)

            value = p.valueWithSamples(fixedSamples)
            unless (value-expected).abs <= 1.0e-10
                assert_fail(<<-MESSAGE

    In the sixth batch of cases:
        calculated value: #{value}
        stored value:     #{expected}

                    MESSAGE
                )
            end

			tol = p.errorEstimate/value
			tol = [tol/2.0,minimumTol].min
			value = p.value(tol)
			accuracy = p.errorEstimate/value
            unless accuracy <= tol
                assert_fail(<<-MESSAGE

    In the sixth batch of cases:
        achieved accuracy: #{accuracy}
        requested:         #{tol}

                    MESSAGE
                )
            end
        }
    end
end

if $0 == __FILE__
    RUNIT::CUI::TestRunner.run(MonteCarloPricerTest.suite)
end

