"""
 Copyright (C) 2000, 2001, 2002 RiskMap srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
"""

__version__ = "$Revision: 1.28 $"
# $Source: /cvsroot/quantlib/QuantLib-Python/QuantLib/test/montecarlo_pricers.py,v $

import QuantLib
import unittest

class MonteCarloPricerTest(unittest.TestCase):
    def runTest(self):
        "Testing Monte Carlo pricers"
        seed = 3456789
        fixedSamples = 100
        minimumTol = 0.01


        # data from "Implementing Derivatives Model", Clewlow, Strickland, pag.118-123
        cases1 = [
            [QuantLib.DiscreteGeometricAPO,  "Call",100.0,100.0, 0.03,0.06,[0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0], 0.2, 5.34255485619]
           ]

        for (pricer, optionType, underlying, strike, dividendYield, riskFreeRate, timeIncrements, volatility, storedValue) in cases1:
            p = pricer(optionType, underlying, strike, dividendYield,
                riskFreeRate, timeIncrements, volatility)
            pvalue = p.value()

            if not (abs(pvalue-storedValue) <= 1e-10):
                self.fail("""
calculated value: %(pvalue)g
stored value:     %(storedValue)g
                          """ % locals())

        # data from "Option Pricing Formulas", Haug, pag.96-97
        cases2 = [
            [QuantLib.EuropeanOption,         "Put", 80.0, 85.0, -0.03, 0.05, 0.25, 0.2, 5.21858890396],
            [QuantLib.ContinuousGeometricAPO, "Put", 80.0, 85.0, -0.03, 0.05, 0.25, 0.2, 4.69221973405]
           ]

        for (pricer, optionType, underlying, strike, dividendYield, riskFreeRate, residualTime, volatility, storedValue) in cases2:
            p = pricer(optionType, underlying, strike, dividendYield,
                riskFreeRate, residualTime, volatility)
            pvalue = p.value()
            if not (abs(pvalue-storedValue) <= 1e-10):
                self.fail("""
calculated value: %(pvalue)g
stored value:     %(storedValue)g
                          """ % locals())


        # trying to approximate the continous version with the discrete version
        cases3 = [
            [QuantLib.DiscreteGeometricAPO, "Put", 80.0, 85.0, -0.03, 0.05, 0.25, 90000, 0.2, 4.6922231469]
           ]

        for (pricer, optionType, underlying, strike, dividendYield, riskFreeRate, residualTime, timesteps, volatility, storedValue) in cases3:
            timeIncrements=[]
            dt=residualTime/timesteps
            for i in range(timesteps):
                timeIncrements.append((i+1)*dt)
            p = pricer(optionType, underlying, strike, dividendYield,
                riskFreeRate, timeIncrements, volatility)
            pvalue = p.value()
            if not (abs(pvalue-storedValue) <= 1e-10):
                self.fail("""
calculated value: %(pvalue)g
stored value:     %(storedValue)g
                          """ % locals())


        cases4 = [
            [QuantLib.McEuropean,      "Put", 80.0, 85.0, -0.03, 0.05, 0.25, 0.2, 5.9135872358, 0],
            [QuantLib.McEuropean,      "Put", 80.0, 85.0, -0.03, 0.05, 0.25, 0.2, 5.42005964479, 1],
            [QuantLib.McEuropean,     "Call", 80.0, 85.0, -0.03, 0.05, 0.25, 0.2, 1.98816310759, 0],
            [QuantLib.McEuropean,     "Call", 80.0, 85.0, -0.03, 0.05, 0.25, 0.2, 2.12098432917, 1],
            [QuantLib.McEuropean, "Straddle", 80.0, 85.0, -0.03, 0.05, 0.25, 0.2, 7.90175034339, 0],
            [QuantLib.McEuropean, "Straddle", 80.0, 85.0, -0.03, 0.05, 0.25, 0.2, 7.54104397396, 1]
           ]
        for (pricer, optionType, underlying, strike, dividendYield, riskFreeRate, residualTime, volatility, storedValue, antithetic) in cases4:
            p = pricer(optionType, underlying, strike, dividendYield,
                riskFreeRate, residualTime, volatility, antithetic,
                seed=seed)
            pvalue = p.valueWithSamples(fixedSamples)
            if not (abs(pvalue-storedValue) <= 1e-10):
                self.fail("""
calculated value: %(pvalue)g
stored value:     %(storedValue)g
                          """ % locals())
            tol = p.errorEstimate()/pvalue
            tol = min(tol/2.0, minimumTol)
            pvalue = p.value(tol)
            accuracy = p.errorEstimate()/pvalue
            if not (accuracy <= tol):
                self.fail("""
accuracy reached   : %(accuracy)g
tolerance requested: %(tol)g
                          """ % locals())


        # data from "Asian Option", Levy, 1997
        # in "Exotic Options: The State of the Art", edited by Clewlow, Strickland
        cases5 = [
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,      0.0, 11.0/12.0,    2, 0.13, 1.38418414762, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,      0.0, 11.0/12.0,    4, 0.13, 1.57691714387, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,      0.0, 11.0/12.0,    8, 0.13, 1.66062743445, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,      0.0, 11.0/12.0,   12, 0.13, 1.68847081883, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,      0.0, 11.0/12.0,   26, 0.13, 1.72955964448, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,      0.0, 11.0/12.0,   52, 0.13, 1.73372169316, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,      0.0, 11.0/12.0,  100, 0.13, 1.74918801089, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,      0.0, 11.0/12.0,  250, 0.13, 1.75421310915, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,      0.0, 11.0/12.0,  500, 0.13, 1.75158383443, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025,      0.0, 11.0/12.0, 1000, 0.13, 1.7516211018 , 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025, 1.0/12.0, 11.0/12.0,    2, 0.13, 1.83665087164, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025, 1.0/12.0, 11.0/12.0,    4, 0.13, 2.00560271429, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025, 1.0/12.0, 11.0/12.0,    8, 0.13, 2.07789721712, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025, 1.0/12.0, 11.0/12.0,   12, 0.13, 2.09622556625, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025, 1.0/12.0, 11.0/12.0,   26, 0.13, 2.14229795212, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025, 1.0/12.0, 11.0/12.0,   52, 0.13, 2.14470270916, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025, 1.0/12.0, 11.0/12.0,  100, 0.13, 2.15954145741, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025, 1.0/12.0, 11.0/12.0,  250, 0.13, 2.16007690017, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025, 1.0/12.0, 11.0/12.0,  500, 0.13, 2.159867044  , 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025, 1.0/12.0, 11.0/12.0, 1000, 0.13, 2.15951634387, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025, 3.0/12.0, 11.0/12.0,    2, 0.13, 2.63315092584, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025, 3.0/12.0, 11.0/12.0,    4, 0.13, 2.76723962361, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025, 3.0/12.0, 11.0/12.0,    8, 0.13, 2.83124836881, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025, 3.0/12.0, 11.0/12.0,   12, 0.13, 2.84290301412, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025, 3.0/12.0, 11.0/12.0,   26, 0.13, 2.88179560417, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025, 3.0/12.0, 11.0/12.0,   52, 0.13, 2.88447044543, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025, 3.0/12.0, 11.0/12.0,  100, 0.13, 2.89985329603, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025, 3.0/12.0, 11.0/12.0,  250, 0.13, 2.90047296063, 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025, 3.0/12.0, 11.0/12.0,  500, 0.13, 2.8981341216 , 1, 1],
            [QuantLib.McDiscreteArithmeticAPO, "Put", 90.0, 87.0, 0.06, 0.025, 3.0/12.0, 11.0/12.0, 1000, 0.13, 2.89703362437, 1, 1]
           ]
        for (pricer, optionType, underlying, strike, dividendYield, riskFreeRate, first, length, fixings, volatility, storedValue, antithetic, controlVariate) in cases5:
            timeIncrements=[first]
            dt=(length)/(fixings-1)
            for i in range(fixings-1):
                timeIncrements.append((i+1)*dt+first)
            p = pricer(optionType, underlying, strike, dividendYield,
                riskFreeRate, timeIncrements, volatility, antithetic, controlVariate,
                seed=seed)
            pvalue = p.valueWithSamples(fixedSamples)
            if not (abs(pvalue-storedValue) <= 1e-10):
                self.fail("""
calculated value: %(pvalue)g
stored value:     %(storedValue)g
                          """ % locals())
            tol = p.errorEstimate()/pvalue
            tol = min(tol/2.0, minimumTol)
            pvalue = p.value(tol)
            accuracy = p.errorEstimate()/pvalue
            if not (accuracy <= tol):
                self.fail("""
accuracy reached   : %(accuracy)g
tolerance requested: %(tol)g
                          """ % locals())


        # data from "Asian Option", Levy, 1997
        # in "Exotic Options: The State of the Art", edited by Clewlow, Strickland
        cases6 = [
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,      0.0, 11.0/12.0,    2, 0.13, 1.51917595129, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,      0.0, 11.0/12.0,    4, 0.13, 1.67940165674, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,      0.0, 11.0/12.0,    8, 0.13, 1.75371215251, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,      0.0, 11.0/12.0,   12, 0.13, 1.77595318693, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,      0.0, 11.0/12.0,   26, 0.13, 1.8143053663 , 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,      0.0, 11.0/12.0,   52, 0.13, 1.82269246898, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,      0.0, 11.0/12.0,  100, 0.13, 1.83822402464, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,      0.0, 11.0/12.0,  250, 0.13, 1.83875059026, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,      0.0, 11.0/12.0,  500, 0.13, 1.83750703638, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025,      0.0, 11.0/12.0, 1000, 0.13, 1.83887181884, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025, 1.0/12.0, 11.0/12.0,    2, 0.13, 1.51154400089, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025, 1.0/12.0, 11.0/12.0,    4, 0.13, 1.67103508506, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025, 1.0/12.0, 11.0/12.0,    8, 0.13, 1.7452968407 , 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025, 1.0/12.0, 11.0/12.0,   12, 0.13, 1.76667074564, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025, 1.0/12.0, 11.0/12.0,   26, 0.13, 1.80528400613, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025, 1.0/12.0, 11.0/12.0,   52, 0.13, 1.81400883891, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025, 1.0/12.0, 11.0/12.0,  100, 0.13, 1.82922901451, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025, 1.0/12.0, 11.0/12.0,  250, 0.13, 1.82937111773, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025, 1.0/12.0, 11.0/12.0,  500, 0.13, 1.82826193186, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025, 1.0/12.0, 11.0/12.0, 1000, 0.13, 1.82967846654, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025, 3.0/12.0, 11.0/12.0,    2, 0.13, 1.49648170891, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025, 3.0/12.0, 11.0/12.0,    4, 0.13, 1.65443100462, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025, 3.0/12.0, 11.0/12.0,    8, 0.13, 1.72817806731, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025, 3.0/12.0, 11.0/12.0,   12, 0.13, 1.74877367895, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025, 3.0/12.0, 11.0/12.0,   26, 0.13, 1.78733801988, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025, 3.0/12.0, 11.0/12.0,   52, 0.13, 1.79624826757, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025, 3.0/12.0, 11.0/12.0,  100, 0.13, 1.81114186876, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025, 3.0/12.0, 11.0/12.0,  250, 0.13, 1.81101152587, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025, 3.0/12.0, 11.0/12.0,  500, 0.13, 1.81002311939, 1, 1],
            [QuantLib.McDiscreteArithmeticASO, "Call", 90.0, 87.0, 0.06, 0.025, 3.0/12.0, 11.0/12.0, 1000, 0.13, 1.81145760308, 1, 1]
           ]
        for (pricer, optionType, underlying, strike, dividendYield, riskFreeRate, first, length, fixings, volatility, storedValue, antithetic, controlVariate) in cases6:
            timeIncrements=[first]
            dt=(length)/(fixings-1)
            for i in range(fixings-1):
                timeIncrements.append((i+1)*dt+first)
            p = pricer(optionType, underlying, dividendYield,
                riskFreeRate, timeIncrements, volatility, antithetic, controlVariate,
                seed=seed)
            pvalue = p.valueWithSamples(fixedSamples)
            if not (abs(pvalue-storedValue) <= 1e-10):
                self.fail("""
calculated value: %(pvalue)g
stored value:     %(storedValue)g
                          """ % locals())
            tol = p.errorEstimate()/pvalue
            tol = min(tol/2.0, minimumTol)
            pvalue = p.value(tol)
            accuracy = p.errorEstimate()/pvalue
            if not (accuracy <= tol):
                self.fail("""
accuracy reached   : %(accuracy)g
tolerance requested: %(tol)g
                          """ % locals())



if __name__ == '__main__':
    print 'testing QuantLib', QuantLib.__version__, QuantLib.QuantLibc.__file__, QuantLib.__file__
    import sys
    suite = unittest.TestSuite()
    suite.addTest(MonteCarloPricerTest())
    if sys.hexversion >= 0x020100f0:
        unittest.TextTestRunner(verbosity=2).run(suite)
    else:
        unittest.TextTestRunner().run(suite)
    raw_input('press any key to continue')

