#ifndef VISUAL_GLCONTEXT_H
#define VISUAL_GLCONTEXT_H

// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.

#include "cvisual.h"
#include <vector>
#include <utility>

namespace visual {
/*
 * These classes are cannonically defined here.  The actual implementations
 * are in the wgl, xgl, agl, and soon, the xgl2 .h and .cpp files. 
 */

/* New model:
 * To actually render a bit of text, the client will create a layout object
 * from the text to be rendered.  All precomputation for the rendering
 * is performed at that time, in a platform-specific way.  This may include
 * rasterizing the font to a temporary array and uploading the array to texture
 * memory.  Every part of the state of the text run that depends only on the
 * text itself is computed.
 * 
 * The client then renders the layout object as-needed.
 * 
 * Layout objects are dynamically allocated by the glFont as-needed.  The only
 * thing that the client can do is get the extent of the layout and render it,
 * both of which are "const" operations.
 */
#define USE_NEWFONT 0

#if USE_NEWFONT
struct glFont
{
	struct layout
	{
		virtual ~layout() = 0;
		// Render the run or paragraph of text, such that in GL coordinates, the
		// center of the bounding box is at <0, 0, 0>, and the text is rendered
		// in the x-y plane. 
		virtual void gl_render() const = 0;
		// Returns the size of the bounding box in the x-y plane for the layed
		// out run of text.
		virtual vector extent() const = 0;
	};
	virtual ~glFont();
	// Compute a layout for a run of text, including any newlines.  The text is
	// always left-justified.
	virtual boost::shared_ptr<layout> lay_out( const std::string&) = 0;
};

#else 
struct glFont 
{
	virtual double getWidth(const char* string) = 0;
	// returns the horizontal extent of the given string in
	//   viewport coordinates (0.0 to 1.0)

	virtual double ascent() = 0;
	// returns the maximum distance from the baseline to the top
	//   of a glyph (e.g. "M")

	virtual double descent() = 0;
	// returns the maximum distance from the baseline to the bottom
	//   of a glyph (e.g. "g")

	virtual void draw(const char* string) = 0;
	// draws string with the current glRasterPos at its left baseline

	virtual void release() = 0;
	// call once for each call to glContext::getFont()
	
	// Defined in glDevice.cpp
	virtual ~glFont();
};

#endif

struct glContext 
{
public:
	virtual void lockMouse() = 0;
	virtual void unlockMouse() = 0;
	virtual int  getMouseButtons() = 0;
	virtual int  getMouseButtonsChanged() = 0;
	virtual int  getShiftKey() = 0;
	virtual int  getAltKey() = 0;
	virtual int  getCtrlKey() = 0;
	virtual vector  getMouseDelta() = 0;
	virtual vector  getMousePos() = 0;
	virtual std::string  getKeys() = 0;

	glContext() {};
	virtual ~glContext() {};

	virtual bool initWindow( const char* title, int x, int y, int width, int height, int flags ) = 0;
	virtual bool changeWindow( const char* title, int x, int y, int width, int height, int flags ) = 0;
	virtual bool isOpen() = 0;
	virtual void cleanup() = 0;
  
	virtual void makeCurrent() = 0;
	virtual void makeNotCurrent() = 0;
	virtual void swapBuffers() = 0;

	virtual vector origin() = 0;   // of entire window (or, same as initWindow())
	virtual vector corner() = 0;   // of entire window (or, same as initWindow())
	virtual int width() = 0;       // of GL area
	virtual int height() = 0;      // of GL area

	virtual std::string lastError() = 0;

	#if !USE_NEWFONT
	virtual glFont* getFont(const char* description, double size) = 0;
	// xxx need to document parameters!
	#else
	virtual boost::shared_ptr<glFont> getFont( double size) = 0;
	#endif
		
	enum {
		DEFAULT    = 0,
		FULLSCREEN = 0x1,
		QB_STEREO  = 0x2
	} WindowFlags;
	
	void add_pending_glDeleteList(int base, int howmany);
	
 private:
    // This is a list of glDeleteLists calls that should be made the next time
    // the context is made active.
    mutex list_lock;
	std::vector<std::pair<int, int> > pending_glDeleteLists;

 protected:
	// Implementors of this class should call this function in their implementation
	// of makeCurrent();
 	void delete_pending_lists();
};

} // !namespace visual

#endif // !VISUAL_GLCONTEXT_H
