/*
 Escape Quotes written by Mike Olson
 Copyright (c) 2001 Fourthought, Inc. USA.   All Rights Reserved.
 See  http://4suite.org/COPYRIGHT  for license and copyright information
*/

#include "Python.h"

static char escape__doc__[] = 
"escape(string) -> string\n\
\n\
Replace all ' with \\' and \\ with \\\\.";


/*
  We need to split the string into a list of strings.  Each of the strings should be ~6000 chars long
  , be UTF-8 encoded, and escaped.  For escaping we use:
  ' --> \'
  \ --> \\\\
  0x00 --> \\000
*/


static PyObject *escape(PyObject *self, PyObject *args) {
  PyObject *str, *result;
  char *orig;
  int size, i, add;

  if (!PyArg_ParseTuple(args, "S:escape", &str)) {
    return NULL;
  }

  /* First pass, determine how characters need to be added */
  orig = PyString_AS_STRING(str);
  size = PyString_GET_SIZE(str);
  for (i = 0, add = 0; i < size; i++) {
    switch (orig[i]) {
    case '\'':
      add += 1;
      break;
    case '\\':
      add += 3;
      break;
    case '\0':
      add += 4;
      break;
    }
  }

  /* Nothing to escape, return original string */
  if (add == 0) {
    Py_INCREF(str);
    return str;
  }

  /* Second pass, replace characters */
  result = PyString_FromStringAndSize(NULL, (size + add));
  if (result) {
    char *p = PyString_AS_STRING(result);
    for (i = 0; i < size; i++) {
      switch (orig[i]) {
      case '\'':
        *p++ = '\\';
        *p++ = '\'';
        add--;
        break;
      case '\\':
        *p++ = '\\';
        *p++ = '\\';
        *p++ = '\\';
        *p++ = '\\';
        add -= 3;
        break;
      case '\0':
        *p++ = '\\';
        *p++ = '\\';
        *p++ = '0';
        *p++ = '0';
        *p++ = '0';
        add -= 4;
        break;
      default:
        *p++ = orig[i];
        continue;
      }
      if (add <= 0) {
        /* copy remaining part */
        i++;
        memcpy(p, orig+i, size-i);
        break;
      }
    }
  }
  return result;
}


static char unescape__doc__[] = 
"unescape(string) -> string\n\
\n\
Convert PostgreSQL escape sequences into normal characters.";

static PyObject *unescape(PyObject *self, PyObject *args) {
  PyObject *str, *result;
  char *orig;
  int size, i, remove;
  
  if (!PyArg_ParseTuple(args, "S:unescape", &str)) {
    return NULL;
  }

  /* First pass, determine how characters need to be removed */
  orig = PyString_AS_STRING(str);
  size = PyString_GET_SIZE(str);
  for (i = 0, remove = 0; i < size;) {
    if (orig[i++] == '\\') {
      if (orig[i] == '\\') {
        remove++;
        i++;
      } else {
        /* The next three are an octect */
        remove += 3;
        i += 3;
      }
    }
  }

  /* Nothing to unescape, return original string */
  if (remove == 0) {
    Py_INCREF(str);
    return str;
  }

  /* Second pass, replace characters */
  result = PyString_FromStringAndSize(NULL, (size - remove));
  if (result) {
    char *p = PyString_AS_STRING(result);
    for (i = 0; i < size; i++) {
      char c = orig[i];
      if (c == '\\') {
        c = orig[++i];
        if (c == '\\') {
          *p++ = c;
          remove--;
        } else {
          /* The next three are an octect */
          c -= '0';
          c = (c << 3) + (orig[++i] - '0');
          c = (c << 3) + (orig[++i] - '0');
          *p++ = c;
          remove -= 3;
        }
        if (remove <= 0) {
          /* copy remaining part */
          i++;
          memcpy(p, orig+i, size-i);
          break;
        }
      } else {
        *p++ = c;
      }
    }
  }
  return result;
}


static PyMethodDef escapeMethods[] = {
  { "escape",   escape,   METH_VARARGS, escape__doc__ },
  { "unescape", unescape, METH_VARARGS, unescape__doc__ },
  { NULL, NULL }
};

DL_EXPORT(void) initEscapec(void) {
  PyObject *module, *dict;
  module = Py_InitModule("Escapec", escapeMethods);
  dict = PyModule_GetDict(module);
}
