#!/usr/bin/env python
########################################################################
#
# File Name:            4xslt.py
#
# Documentation:        http://docs.4suite.org/4XSLT/4xslt.py.html
#
"""
Command-line invokation of the 4XSLT processor
WWW: http://4suite.org/4XSLT        e-mail: support@4suite.org

Copyright (c) 1999-2001 Fourthought, Inc., USA.   All Rights Reserved.
See  http://4suite.org/COPYRIGHT  for license and copyright information
"""

import re, string, os, sys, getopt, cStringIO, traceback, urllib
import xml.dom.ext
from xml import xpath
from xml.xslt import XsltException
from xml.xslt import Processor
from Ft import __version__
from xml.xslt.Debugger import DebugController

__doc__ = """4XSLT Command line application."""

g_paramBindingPattern = re.compile(r"([\d\D_\.\-]*:?[\d\D_\.\-]+)=(.*)")

MAX_PYTHON_RECURSION_DEPTH=10000

if sys.hexversion >= 0x2000000:
    sys.setrecursionlimit(MAX_PYTHON_RECURSION_DEPTH)

def Run(options,args):

    if options.has_key('version'):
        print '4XSLT version %s' % __version__
        return

    defs = options.get('define', [])
    if type(defs) != type([]):
        defs = [defs]

    include_paths = options.get('includepath', [])
    if type(include_paths) != type([]):
        include_paths = [include_paths]

    top_level_params = {}
    for d in defs:
        match = g_paramBindingPattern.match(d)
        top_level_params[match.group(1)] = match.group(2)


    #Instead of renaming this through the code, I'm just gonna define the correct variables here
    xinclude = not options.has_key('noxinclude')
    ignore_pis = options.has_key('ignore')
    out_file = options.has_key('outfile') and open(options['outfile'], 'w') or sys.stdout
    validate_flag = options.has_key('validate')

    # Do the file checking here; the rest of 4XSLT is uri based
    source = args['source-uri']
    if os.path.isfile(source):
        source = urllib.pathname2url(source)

    stylesheets = []
    for uri in args['stylesheet-uri']:
        if os.path.isfile(uri):
            uri = urllib.pathname2url(uri)
        stylesheets.append(uri)

    stacktrace_on_error = options.has_key('stacktrace-on-error')

    if options.has_key('trace') or options.has_key('debug'):

        if not options.has_key('outfile'):
            raise SystemExit("If you are using the trace, debug, or profile options, you must specify an output file for the XSLT transform results")
        
        from xml.xslt import ExtendedProcessingElements, StylesheetReader
        processor = ExtendedProcessingElements.ExtendedProcessor(stylesheetIncPaths=include_paths)
        StylesheetReader.g_mappings = ExtendedProcessingElements.GetMappings()
        if options.has_key('trace'):
            processor._4xslt_trace = 1
            if options.has_key('trace-file'):
                processor._4xslt_traceStream = open(options['trace-file'],'w')
        if options.has_key('debug'):
            processor._4xslt_debug = 1

            processor._ft_debug_controller = DebugController.DebugController()

    else:
        processor = Processor.Processor(stylesheetIncPaths=include_paths)

    from Ft.Lib import pDomlette
    BETA_DOMLETTE = int(os.environ.get("BETA_DOMLETTE", 0))
    if BETA_DOMLETTE and not validate_flag:
        from Ft.Lib import cDomlette
        g_readerClass = cDomlette.RawExpatReader
        reader = cDomlette.RawExpatReader()
    elif validate_flag:
        reader = pDomlette.SaxReader(validate=1)
    else:
        reader = pDomlette.PyExpatReader(processIncludes=xinclude)

    try:
        processor.setDocumentReader(reader)
        for sty in stylesheets:
            processor.appendStylesheetUri(sty)
        if source == '-':
            processor.runStream(sys.stdin, ignore_pis,
                                topLevelParams=top_level_params,
                                outputStream=out_file)
        else:
            processor.runUri(source, ignore_pis,
                             topLevelParams=top_level_params,
                             outputStream=out_file)
    except XsltException, e:
        if stacktrace_on_error:
            s = cStringIO.StringIO()
            traceback.print_exc(1000, s)
            sys.stderr.write(s.getvalue())
        sys.stderr.write(str(e) + '\n')
        sys.exit(-1)
    except (xpath.RuntimeException, xpath.CompiletimeException), e:
        if stacktrace_on_error:
            s = cStringIO.StringIO()
            traceback.print_exc(1000, s)
            sys.stderr.write(s.getvalue())
        if hasattr(e, 'stylesheetUri'):
            sys.stderr.write("While processing %s\n" % e.stylesheetUri)
        sys.stderr.write(str(e) + '\n')
        sys.exit(-1)
    out_file.close()

from Ft.Lib.CommandLine import Options, CommandLineApp, Arguments, Command


class XsltCommandLineApp(CommandLineApp.CommandLineApp):
    def __init__(self):
        CommandLineApp.CommandLineApp.__init__(
            self,
            '4xslt',
            '4XSLT version %s' % __version__,
            __doc__,
            [],
            ourOptions = Options.Options([Options.Option('V',
                                                         'version',
                                                         'Display current version.',
                                                         ),
                                          Options.Option('v',
                                                         'validate',
                                                         'Validate the input file as it is being parsed.',
                                                         ),
                                          Options.Option('i',
                                                         'ignore',
                                                         'Ignore stylesheet processing instructions '\
                                                         'in the input file.',
                                                         ),
                                          Options.Option('D',
                                                         'define=definition',
                                                         'Bind a top-level parameter, overriding any '\
                                                         'binding in the stylesheet.  A definition is '\
                                                         'in the form of name=value',
                                                         ),
                                          Options.Option('I',
                                                         'includepath=baseUri',
                                                         'Set a base URI where stylesheet import and include will search'
                                                         ),
                                          Options.Option('o',
                                                         'outfile=fileName',
                                                         'Specify a filename for the output.  This file '\
                                                         'will be overwritten if present.',
                                                         ),
                                          Options.Option('e',
                                                         'stacktrace-on-error',
                                                         'Display a stack trace trace when an error occurs.',
                                                         ),
                                          Options.Option(None,
                                                         'noxinclude',
                                                         'Do no expansions of XInclusions in source documents or stylesheets.',
                                                         ),
                                          Options.Option(None,
                                                         'trace',
                                                         'Output information tracing the progress the processing as it  occurs.          Output will be sent to standard output by default.',
                                                         ),
                                          Options.Option(None,
                                                         'trace-file=',
                                                         'Specify output file fof execution trace output..',
                                                         ),
                                          Options.Option('d',
                                                         'debug',
                                                         'Execute 4xslt in the command line debugger.',
                                                         ),
                                          ]),
            enableShowCommands = 0
            )

        self.function = Run
        self.arguments = [Arguments.RequiredArgument(
            'source-uri',
            'The URI of the source XML document.  Note: if you use "-" as the name'\
            'of the source document, the source will  instead be read from standard input.',
            str),
                          Arguments.ZeroOrMoreArgument('stylesheet-uri',
                                                      'The URI(s) of the stylesheets to apply to the source XML.',
                                                      str),
                          ]


    def validate_arguments(self,args):
        return Command.Command.validate_arguments(self,args)



def Register():
    
    return XsltCommandLineApp()
