/* ToolBarButtonSupport.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.actions;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.ButtonModel;
import javax.swing.Icon;

/**
 * Provides some common methods for {@link ToolBarButton}
 * and {@link ToolBarToggleButton}.
 */
final class ToolBarButtonSupport {

    // Clients should not instantiate this class
    private ToolBarButtonSupport() {
    }

    /**
     * Configure properties from actions.
     */
    static void configurePropertiesFromAction(AbstractButton b, Action a) {
        b.setText(null);

        String tooltiptext = (String) a.getValue(Action.SHORT_DESCRIPTION);
        if (tooltiptext == null) {
            tooltiptext = (String) a.getValue(Action.NAME);
        }
        b.setToolTipText(tooltiptext);

        Icon icon = (Icon) a.getValue(SimpleAction.MEDIUM_ICON);
        if (icon == null) {
            icon = (Icon) a.getValue(Action.SMALL_ICON);
        }
        b.setIcon(icon);

        Icon disabledIcon = (Icon) a.getValue(SimpleAction.MEDIUM_DISABLED_ICON);
        b.setDisabledIcon(disabledIcon);

        String actionCommand = (String) a.getValue(Action.ACTION_COMMAND_KEY);
        b.setActionCommand(actionCommand);

        b.setEnabled(a.isEnabled());
    }
    
    /**
     * Configure model from actions.
     */
    static void configureModelFromAction(AbstractButton b, Action a) {        
        ButtonModel buttonModel = (ButtonModel) a.getValue(SimpleAction.TOGGLE_BUTTON_MODEL);
        if (buttonModel != null) {
            b.setModel(buttonModel);
        }
    }

    /**
     * Create an action property change listener for the given button.
     */
    static PropertyChangeListener createActionPropertyChangeListener(AbstractButton b) {
        return new ActionPropertyChangeListener(b);
    }

    /**
     * Property change listener which keeps appearance of button in sync with
     * the action.
     */
    private static class ActionPropertyChangeListener
            implements PropertyChangeListener {

        //
        private final AbstractButton b;

        //
        ActionPropertyChangeListener(AbstractButton b) {
            this.b = b;
        }

        //
        public void propertyChange(PropertyChangeEvent evt) {
            if ("enabled".equals(evt.getPropertyName())) {
                b.setEnabled(((Boolean) evt.getNewValue()).booleanValue());
            } else {
                configurePropertiesFromAction(b, (Action) evt.getSource());
            }
            b.invalidate();
            b.repaint();
        }
    }
}
