/*
 *  NachoCalendar
 *
 * Project Info:  http://nachocalendar.sf.net
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 * Changes
 * -------
 *
 * 2004-10-22   setEnabled(boolean b) overriden, now works
 * 2004-10-18   Added previousMonth and nextMonth methods
 * 2004-10-01   Checked with checkstyle
 * 2004-9-3     Fixed borders
 *
 * -------
 *
 * MonthScroller.java
 *
 * Created on August 12, 2004, 6:47 PM
 */

package net.sf.nachocalendar.components;

import java.text.DateFormatSymbols;

import javax.swing.DefaultComboBoxModel;
import javax.swing.Icon;
import javax.swing.SwingConstants;
import javax.swing.event.ChangeEvent;

import net.sf.nachocalendar.event.YearChangeEvent;

/**
 * Component used to change the current month. It extends from JComboBox
 * and has two arrows for unitary changes. It fires a YearChangeEvent
 * when the month under or overflows. It also fires a ChangeEvent when
 * the month is changed.
 * @author Ignacio Merani
 */
public class MonthScroller extends javax.swing.JPanel {
    
    /** Creates new form MonthScroller. */
    public MonthScroller() {
        initComponents();
        DateFormatSymbols symbols = FormatSymbols.getSingletonObject(); //get the formatter
        String[] months = symbols.getMonths(); //this returns the text form of months
        for (int i = 0; i < months.length - 1; i++) {
            months[i] = months[i].substring(0, 1).toUpperCase() + months[i].substring(1).toLowerCase();
        }
        cMonths.setModel(new DefaultComboBoxModel(months));
        cMonths.removeItemAt(cMonths.getItemCount() - 1);
    }
    
    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;
        
        bPrevious = new ArrowButton(SwingConstants.WEST);
        cMonths = new javax.swing.JComboBox();
        bNext = new ArrowButton(SwingConstants.EAST);
        
        FormListener formListener = new FormListener();
        
        setLayout(new java.awt.GridBagLayout());
        
        setBorder(new javax.swing.border.EtchedBorder());
        bPrevious.addActionListener(formListener);
        
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weighty = 1.0;
        add(bPrevious, gridBagConstraints);
        
        cMonths.addActionListener(formListener);
        
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        add(cMonths, gridBagConstraints);
        
        bNext.addActionListener(formListener);
        
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weighty = 1.0;
        add(bNext, gridBagConstraints);
        
    }
    
    private class FormListener implements java.awt.event.ActionListener {
        public void actionPerformed(java.awt.event.ActionEvent evt) {
            if (evt.getSource() == bPrevious) {
                MonthScroller.this.bPreviousActionPerformed(evt);
            } else if (evt.getSource() == cMonths) {
                MonthScroller.this.cMonthsActionPerformed(evt);
            } else if (evt.getSource() == bNext) {
                MonthScroller.this.bNextActionPerformed(evt);
            }
        }
    }
    
    private void cMonthsActionPerformed(java.awt.event.ActionEvent evt) {
        fireChangeListenerStateChanged(new ChangeEvent(this));
    }
    
    /**
     * Changes to the next month.
     */
    public void nextMonth() {
        int month = getMonth();
        if (month == 11) {
            setMonth(0);
            fireYearChangeListenerYearIncreased(new YearChangeEvent(this));
        } else {
            setMonth(month + 1);
        }
        fireChangeListenerStateChanged(new ChangeEvent(this));
    }
    
    /**
     * Changes to the previous month.
     */
    public void previousMonth() {
        int month = getMonth();
        if (month == 0) {
            setMonth(11);
            fireYearChangeListenerYearDecreased(new YearChangeEvent(this));
        } else {
            setMonth(month - 1);
        }
        fireChangeListenerStateChanged(new ChangeEvent(this));
    }
    
    private void bNextActionPerformed(java.awt.event.ActionEvent evt) {
        nextMonth();
    }
    
    private void bPreviousActionPerformed(java.awt.event.ActionEvent evt) {
        previousMonth();
    }
    
    /**
     * Changes de current month.
     * @param month new month
     */
    public void setMonth(int month) {
        cMonths.setSelectedIndex(month);
    }
    
    /**
     * Returns the current month.
     * @return current month
     */
    public int getMonth() {
        return cMonths.getSelectedIndex();
    }
    
    /**
     * Changes the icon for the Next button.
     * @param icon new Icon or null to show the default arrow
     */
    public void setNextIcon(Icon icon) {
        bNext.setIcon(icon);
    }
    
    /**
     * Changes the icon for the Previous button.
     * @param icon new Icon or null to show the default arrow
     */
    public void setPreviousIcon(Icon icon) {
        bPrevious.setIcon(icon);
    }
    
    /**
     * Changes the text of the Next button.
     * @param text new text or null to show the default arrow
     */
    public void setNextText(String text) {
        bNext.setText(text);
    }
    
    /**
     * Changes the text of the Previous button.
     * @param text new text or null to show the default arrow
     */
    public void setPreviousText(String text) {
        bPrevious.setText(text);
    }
    
    /**
     * Registers YearChangeListener to receive events.
     *
     * @param listener
     *            The listener to register.
     */
    public synchronized void addYearChangeListener(
    net.sf.nachocalendar.event.YearChangeListener listener) {
        if (yearChangeListenerList == null) {
            yearChangeListenerList = new java.util.ArrayList();
        }
        yearChangeListenerList.add(listener);
    }
    
    /**
     * Removes YearChangeListener from the list of listeners.
     *
     * @param listener
     *            The listener to remove.
     */
    public synchronized void removeYearChangeListener(
    net.sf.nachocalendar.event.YearChangeListener listener) {
        if (yearChangeListenerList != null) {
            yearChangeListenerList.remove(listener);
        }
    }
    
    /**
     * Notifies all registered listeners about the event.
     *
     * @param event
     *            The event to be fired
     */
    private void fireYearChangeListenerYearIncreased(YearChangeEvent event) {
        java.util.ArrayList list;
        synchronized (this) {
            if (yearChangeListenerList == null) {
                return;
            }
            list = (java.util.ArrayList) yearChangeListenerList.clone();
        }
        for (int i = 0; i < list.size(); i++) {
            ((net.sf.nachocalendar.event.YearChangeListener) list.get(i))
            .yearIncreased(event);
        }
    }
    
    /**
     * Notifies all registered listeners about the event.
     *
     * @param event
     *            The event to be fired
     */
    private void fireYearChangeListenerYearDecreased(YearChangeEvent event) {
        java.util.ArrayList list;
        synchronized (this) {
            if (yearChangeListenerList == null) {
                return;
            }
            list = (java.util.ArrayList) yearChangeListenerList.clone();
        }
        for (int i = 0; i < list.size(); i++) {
            ((net.sf.nachocalendar.event.YearChangeListener) list.get(i))
            .yearDecreased(event);
        }
    }
    
    /**
     * Registers ChangeListener to receive events.
     *
     * @param listener
     *            The listener to register.
     */
    public synchronized void addChangeListener(
    javax.swing.event.ChangeListener listener) {
        if (changeListenerList == null) {
            changeListenerList = new java.util.ArrayList();
        }
        changeListenerList.add(listener);
    }
    
    /**
     * Removes ChangeListener from the list of listeners.
     *
     * @param listener
     *            The listener to remove.
     */
    public synchronized void removeChangeListener(
    javax.swing.event.ChangeListener listener) {
        if (changeListenerList != null) {
            changeListenerList.remove(listener);
        }
    }
    
    /**
     * Notifies all registered listeners about the event.
     *
     * @param event
     *            The event to be fired
     */
    private void fireChangeListenerStateChanged(
    javax.swing.event.ChangeEvent event) {
        java.util.ArrayList list;
        synchronized (this) {
            if (changeListenerList == null) {
                return;
            }
            list = (java.util.ArrayList) changeListenerList.clone();
        }
        for (int i = 0; i < list.size(); i++) {
            ((javax.swing.event.ChangeListener) list.get(i))
            .stateChanged(event);
        }
    }
    
    /**
     * Enables or disables the component.
     * @param b true for enabling
     */
    public void setEnabled(boolean b) {
        bNext.setEnabled(b);
        bPrevious.setEnabled(b);
        cMonths.setEnabled(b);
    }
    
    /**
     * Getter for enabled property.
     * @return true if it's enabled
     */
    public boolean isEnabled() {
        return cMonths.isEnabled();
    }
    
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton bNext;
    private javax.swing.JButton bPrevious;
    private javax.swing.JComboBox cMonths;
    // End of variables declaration//GEN-END:variables
    
    /**
     * Utility field holding list of YearChangeListeners.
     */
    private transient java.util.ArrayList yearChangeListenerList;
    
    /**
     * Utility field holding list of ChangeListeners.
     */
    private transient java.util.ArrayList changeListenerList;
    
}
