/**
 * GUI Commands
 * Copyright 2004 Andrew Pietsch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: InlineMember.java,v 1.13 2006/02/26 00:59:06 pietschy Exp $
 */

package org.pietschy.command;

import org.pietschy.command.log.Logger;

import javax.swing.*;
import java.util.Iterator;
import java.util.List;


class
InlineMember
extends AbstractGroupMember
implements GroupListener
{
   private static final Logger log = CommandManager.getLogger(InlineMember.class);

   private CommandGroup inlineGroup;
   private CommandGroup parent;

   protected InlineMember(CommandGroup parent, CommandGroup inlineGroup)
   {
      if (!parent.isAllowableMember(inlineGroup))
         throw new IllegalArgumentException("Command: " + inlineGroup +
                                            " is not allowed in group: " + parent);

      this.parent = parent;
      this.inlineGroup = inlineGroup;
      inlineGroup.addGroupListener(this);
   }


   /**
    * Listen to when the group changes and ask our parent to rebuild.
    */
   public void membersChanged(GroupEvent e)
   {
      parent.rebuildAllPopups();
   }

   public boolean isMemberFor(Command c)
   {
      return inlineGroup.equals(c);
   }

   public void addComponentTo(JComponent container, Object factory, String faceId, List previousButtons, int buttonIndex)
   {
      log.enter("addTo");
      log.param("container", String.valueOf(container.getClass()));

      int index = 0;
      for (Iterator iter = inlineGroup.getMemberList().memberIterator(); iter.hasNext();)
      {
         ((GroupMember)iter.next()).addComponentTo(container, factory, faceId, previousButtons, index++);
      }

      log.exit("addTo()");
   }

   /**
    * Checks if this group is dependant on the specified command.
    *
    * @param c the Command in question.
    * @return <tt>true</tt> if this member
    */
   public boolean isDependantOn(Command c)
   {
      String id = inlineGroup.getId();
      if (id != null && id.equals(c.getId()))
         return true;

      return inlineGroup.contains(c);
   }

   public void
   acceptVisitor(GroupVisitor visitor)
   {
      inlineGroup.visitChildren(visitor);
   }

   /**
    * Returns a string representation of the object. In general, the
    * <code>toString</code> method returns a string that
    * "textually represents" this object. The result should
    * be a concise but informative representation that is easy for a
    * person to read.
    * It is recommended that all subclasses override this method.
    * <p/>
    * The <code>toString</code> method for class <code>Object</code>
    * returns a string consisting of the name of the class of which the
    * object is an instance, the at-sign character `<code>@</code>', and
    * the unsigned hexadecimal representation of the hash code of the
    * object. In other words, this method returns a string equal to the
    * value of:
    * <blockquote>
    * <pre>
    * getClass().getName() + '@' + Integer.toHexString(hashCode())
    * </pre></blockquote>
    *
    * @return a string representation of the object.
    */
   public String toString()
   {
      return getClass().getName() + "[" + inlineGroup.getId() + "]";
   }


}
