/*

    Memory allocation wrapper class
    Copyright (C) 2000-2001 Jussi Laako

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <stdlib.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <string>
#include <stdexcept>


#ifndef ALLOC_HH
    #define ALLOC_HH


    using namespace std;


    class clAlloc
    {
        protected:
            bool bLocked;
            long lSize;
            void *vpPtr;
        public:
            clAlloc ()
                {
                    bLocked = false;
                    lSize = 0l;
                    vpPtr = NULL;
                }
            clAlloc (long lAllocSize)
                {
                    bLocked = false;
                    lSize = 0l;
                    vpPtr = NULL;
                    Size(lAllocSize);
                }
            ~clAlloc ()
                { Free(); }
            void *Size (long lAllocSize)
                {
                    Free();
                    lSize = lAllocSize;
                    vpPtr = (void *) malloc(lAllocSize);
                    if (vpPtr == NULL) 
                        throw runtime_error(string("Out of memory!"));
                    return vpPtr;
                }
            void *Resize (long lAllocSize)
                {
                    if (bLocked) UnLock();
                    lSize = lAllocSize;
                    vpPtr = (void *) realloc(vpPtr, lAllocSize);
                    if (vpPtr == NULL)
                        throw runtime_error(string("Out of memory!"));
                    return vpPtr;
                }
            void Free ()
                {
                    if (bLocked) UnLock();
                    if (vpPtr != NULL) 
                    {
                        free(vpPtr);
                        lSize = 0l;
                        vpPtr = NULL;
                    }
                }
            void *GetPtr ()
                { return vpPtr; }
            long GetSize ()
                { return lSize; }
            void Lock ()
                {
                    bLocked = true;
                    mlock(vpPtr, lSize);
                }
            void UnLock ()
                {
                    bLocked = false;
                    munlock(vpPtr, lSize);
                }
            void Copy (const clAlloc &Src)
                {
                    Free();
                    Size(Src.lSize);
                    memcpy(vpPtr, Src.vpPtr, lSize);
                }
            void CopyTo (clAlloc &Dest)
                {
                    Dest.Copy(*this);
                }
            clAlloc GetCopy ()
                {
                    clAlloc Temp;

                    Temp.Copy(*this);
                    return Temp;
                }
            operator char *() 
                { return ((char *) vpPtr); }
            operator unsigned char *() 
                { return ((unsigned char *) vpPtr); }
            operator short *() 
                { return ((short *) vpPtr); }
            operator unsigned short *() 
                { return ((unsigned short *) vpPtr); }
            operator int *()
                { return ((int *) vpPtr); }
            operator unsigned int *()
                { return ((unsigned int *) vpPtr); }
            operator long *()
                { return ((long *) vpPtr); }
            operator unsigned long *()
                { return ((unsigned long *) vpPtr); }
            operator float *()
                { return ((float *) vpPtr); }
            operator double *()
                { return ((double *) vpPtr); }
            operator long double *()
                { return ((long double *) vpPtr); }
            operator void *()
                { return vpPtr; }
            clAlloc & operator = (const clAlloc &Src)
                { 
                    Copy(Src);
                    return (*this);
                }
    };

#endif

