/* GNOME Transcript
 * Copyright (C) 1999-2000 the Free Software Foundation 
 * Authors : Matias Mutchinick
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/**
 * The table editor widget implementation. 
 * To edit the contents of a table.
 */

/**
 * How does it works?
 * The GTransTableEditor is an engine that allows you to modifiy the
 * data of a table. 
 * When you make a change to a row in the GTransTableEditor, it appends a
 * record of that change in a GList.
 * Then if you want to commit the changes made to the server you can
 * use the following functions:
 *
 * gtrans_table_editor_can_next:
 * Returns 1 if there are any more records of changes in the list.
 *
 * gtrans_table_editor_pop_query:
 * Returns the query string of the modification record standing next
 * in line, to create the quey it uses the data in the matrix widget
 * as the new data and the data in the GTransIFaseResult that you
 * passed to create the GTransTableEditor as the old data.
 *
 * gtrans_table_editor_sync:
 * Use this function if you succesfully executed the query string 
 * in the server, this function erases the modification record,
 * and syncronizes the Matrix widget with the GTransIFaseResult 
 * that stands in the back of the editor.
 *
 */



#include <gnome.h>
#include "gtrans_string.h"
#include <gtrans_ifase/gtrans_ifase.h>

#include "gtkmatrix.h"
#include "gtrans_table_editor.h"
#include "gtrans_button.h"


/* Types of modification records */
enum {
	GTE_UPDATE,
	GTE_INSERT,
	GTE_DELETE
};


/* The modification record */
typedef struct _GTE_Record {
	
	/* Index of the row */
	gint      row;

	/* For the previous state */
	gint      prev;

	/* GTE_UPDATE, GTE_INSERT , GTE_DELETE */
	gint      type;

} GTE_Record;





/* Signals ... */
enum {
	COMMIT_ALL,
	COMMIT_NEXT,
	STOP,
	CLOSE,
	LAST_SIGNAL
};



static guint gtrans_table_editor_signals[LAST_SIGNAL] = { 0 };
static GtkWindow *parent_class = NULL;




/**
 *-----------------------------------------------------------------------------
 * Widget stuff
 *-----------------------------------------------------------------------------
 */


/**
 * gtrans_table_editor_destroy
 * @object : The GTransTableEditor 
 *
 * free all the memory used by the GTransTableEditor..
 */
static void
gtrans_table_editor_destroy(GtkObject *object)
{
       	GTransTableEditor *gte;
	GList             *tmp;
	
	g_return_if_fail(object != NULL);
	g_return_if_fail(GTRANS_IS_TABLE_EDITOR(object));
	
	gte = GTRANS_TABLE_EDITOR(object);
	
	if (gte->tname != NULL)
		g_free(gte->tname);
	
	if (gte->mod != NULL){
		for (tmp = gte->mod ; tmp != NULL ; tmp = tmp->next)
			if (tmp->data)
				g_free(tmp->data);
		
		g_list_free(gte->mod);
	}
	
	if (gte->res != NULL)
		gtrans_ifase_result_free(gte->res);
	
	gtk_widget_destroy(gte->row_menu);
	
	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		(*GTK_OBJECT_CLASS(parent_class)->destroy) (object);
}




/**
 * gtrans_table_editor_class_init
 *
 * Widget stuff.
 */
static void
gtrans_table_editor_class_init(GTransTableEditorClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;
	
	object_class = (GtkObjectClass *) class;
	widget_class = (GtkWidgetClass *) class;
	
	gtrans_table_editor_signals[COMMIT_ALL] =
		gtk_signal_new ("commit_all",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GTransTableEditorClass, commit_all),
				gtk_signal_default_marshaller,
				GTK_TYPE_NONE, 0);
	
	gtrans_table_editor_signals[COMMIT_NEXT] =
		gtk_signal_new ("commit_next",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GTransTableEditorClass, commit_next),
				gtk_signal_default_marshaller,
				GTK_TYPE_NONE, 0);
	
	gtrans_table_editor_signals[STOP] =
		gtk_signal_new ("stop",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GTransTableEditorClass, stop),
				gtk_signal_default_marshaller,
				GTK_TYPE_NONE, 0);
	
	gtrans_table_editor_signals[CLOSE] =
		gtk_signal_new ("close",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GTransTableEditorClass, close),
				gtk_signal_default_marshaller,
				GTK_TYPE_NONE, 0);
	
	gtk_object_class_add_signals(object_class, 
				     gtrans_table_editor_signals,
				     LAST_SIGNAL);
	
	parent_class = gtk_type_class(gtk_window_get_type());
	
	/* Destroy */
	object_class->destroy = gtrans_table_editor_destroy;
}




/**
 * gtrans_table_editor_init
 *
 * Initialize some values for the GTransTableEditor.
 */
static void
gtrans_table_editor_init(GTransTableEditor *gte)
{
	gte->mod = NULL;
	gte->flags = 0;
}




/**
 * gtrans_table_editor_get_type
 *
 * Widget stuff
 */
guint 
gtrans_table_editor_get_type()
{
	static guint gtrans_table_editor_type = 0;
	
	if(!gtrans_table_editor_type) {
		GtkTypeInfo gtrans_table_editor_info = {
			"GTransTableEditor",
			sizeof(GTransTableEditor),
			sizeof(GTransTableEditorClass),
			(GtkClassInitFunc) gtrans_table_editor_class_init,
			(GtkObjectInitFunc) gtrans_table_editor_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		
		gtrans_table_editor_type = 
			gtk_type_unique(gtk_window_get_type(),
					&gtrans_table_editor_info);
	}
	return gtrans_table_editor_type;
}



/**
 *-----------------------------------------------------------------------------
 * Signal emmitions...
 *-----------------------------------------------------------------------------
 */

/**
 * gtrans_table_editor_emit__commit_all
 *
 * Emits the "commit_all" signal, called when the "Commit All" 
 * button is clicked.
 */
void
gtrans_table_editor_emit__commit_all(GtkWidget  *widget)
{
	GTRANS_TABLE_EDITOR_SET_FLAGS(widget,GTRANS_TABLE_EDITOR_COMMITING);
	gtk_signal_emit(GTK_OBJECT(widget),gtrans_table_editor_signals[COMMIT_ALL]);
	GTRANS_TABLE_EDITOR_UNSET_FLAGS(widget,GTRANS_TABLE_EDITOR_COMMITING);
}



/**
 * gtrans_table_editor_emit__commit_next
 *
 * Emits the "commit_next" signal, called when the "Commit Next" button is clicked.
 */
void
gtrans_table_editor_emit__commit_next(GtkWidget  *widget)
{
	gtk_signal_emit(GTK_OBJECT(widget),gtrans_table_editor_signals[COMMIT_NEXT]);
}




/**
 * gtrans_table_editor_emit__stop
 *
 * Emits the "stop" signal, called when the "Stop" button is clicked.
 */
void
gtrans_table_editor_emit__stop(GtkWidget  *widget)
{
	if (!GTRANS_TABLE_EDITOR_COMMITING(widget))
		return;
	
	GTRANS_TABLE_EDITOR_SET_FLAGS(widget,GTRANS_TABLE_EDITOR_INTERRUPT);
	gtk_signal_emit(GTK_OBJECT(widget),gtrans_table_editor_signals[STOP]);
}




/**
 * gtrans_table_editor_emit__close
 *
 * Emits the "close" signal, called when the "Close" button is clicked.
 */
void
gtrans_table_editor_emit__close(GtkWidget  *widget)
{
	gtk_signal_emit(GTK_OBJECT(widget),gtrans_table_editor_signals[CLOSE]);
}








/**
 *-----------------------------------------------------------------------------
 * Keeping track of changes made.
 *-----------------------------------------------------------------------------
 */


/**
 * gtrans_table_editor_find_record_link:
 * @gte : The GTransTableEditor
 * @row : Number of a row
 *
 * Return the link of the GList * of records (@gte->mod) matching
 * the row number @row.
 */
static GList *
gtrans_table_editor_find_record_link(GTransTableEditor *gte,
				     gint               row)
{
	GTE_Record  *rec;
	GList       *tmp;
	
	for (tmp = gte->mod ; tmp != NULL ; tmp = tmp->next){
		
		rec = (GTE_Record*)tmp->data;
		if ( rec->row == row )
			return tmp;
	}
	return NULL;
}




/**
 * gtrans_table_editor_find_record:
 * @gte : The GTransTableEditor
 * @row : Number of a row
 *
 * Return the record of the change made to the row number @row.
 */
static GTE_Record *
gtrans_table_editor_find_record(GTransTableEditor *gte,
				gint               row)
{
	GTE_Record  *rec;
	GList       *tmp;
	
	for (tmp = gte->mod ; tmp != NULL ; tmp = tmp->next){
		
		rec = (GTE_Record*)tmp->data;
		if ( rec->row == row )
			return rec;
	}
	return NULL;
}





/**
 * gtrans_table_editor_toggle_row_undel
 * @gte : The GTransTableEditor
 * @row : The index of a row
 *
 * Unmarks the @row previously marked for deletion and
 * sets the row's state back to what it was before
 * it was marked for deletion.
 * All this is handled in the modification record
 * that belongs to this row.
 */
static void
gtrans_table_editor_toggle_row_undel(GTransTableEditor *gte,
				     gint               row)
{
	GTE_Record *rec;
	GList      *link;
	gchar      *label = NULL;
	
	g_return_if_fail(gte != NULL);
	g_return_if_fail(GTK_MATRIX(gte->matrix)->maxrow > row);
	
	link = gtrans_table_editor_find_record_link(gte,row);
	if (link == NULL)
		return;
	
	rec = (GTE_Record*) link->data;
	
	if (rec->type != GTE_DELETE)
		return;
	
	switch (rec->prev){
	case GTE_DELETE:
		gte->mod = g_list_remove_link(gte->mod,link);
		label = g_strdup("");
		break;
		
	case GTE_UPDATE:
		rec->type = GTE_UPDATE;
		label = g_strdup("*");
		break;
		
	case GTE_INSERT:
		rec->type = GTE_INSERT;
		label = g_strdup("+");
		break;
	} 
	
	gtk_matrix_row_button_add_label(GTK_MATRIX(gte->matrix),row,label);
	g_free(label);
}





/**
 * gtrans_table_editor_toggle_row_del
 * @gte : The GTransTableEditor
 * @row : The index of a row
 *
 * Marks the @row for deletion.
 */
static void
gtrans_table_editor_toggle_row_del(GTransTableEditor *gte,
				   gint               row)
{
	GTE_Record *rec;
	gchar      *label;
	
	g_return_if_fail(gte != NULL);
	g_return_if_fail(GTK_MATRIX(gte->matrix)->maxrow > row);
	
	rec = gtrans_table_editor_find_record(gte,row);
	
	if (rec != NULL)
		rec->type = GTE_DELETE;
	else {
		
		rec = g_new(GTE_Record,1);
		
		rec->row  = row;
		rec->prev = GTE_DELETE;
		rec->type = GTE_DELETE;
		
		gte->mod = g_list_append(gte->mod,rec);
	}
	
	label = g_strdup("-");
	gtk_matrix_row_button_add_label(GTK_MATRIX(gte->matrix),row,label);
	g_free(label);
}




/**
 * gtrans_table_editor_toggle_row_new
 */
static void
gtrans_table_editor_toggle_row_new(GTransTableEditor *gte,
				   gint               row)
{
	GTE_Record *rec;
	gchar      *label;
	
	g_return_if_fail(gte != NULL);
	g_return_if_fail(GTK_MATRIX(gte->matrix)->maxrow == row);
	
	rec = g_new(GTE_Record,1);
	
	rec->row  = row;
	rec->prev = GTE_INSERT;
	rec->type = GTE_INSERT;
	
	gte->mod = g_list_append(gte->mod,rec);
	
	label = g_strdup("+");
	gtk_matrix_row_button_add_label(GTK_MATRIX(gte->matrix),row,label);
	g_free(label);
	
	gtk_matrix_add_row(GTK_MATRIX(gte->matrix),1);
}




/**
 * gtrans_table_editor_toggle_row_mod
 */
static void
gtrans_table_editor_toggle_row_mod(GTransTableEditor *gte,
				   gint               row)
{
	GTE_Record *rec;
	gchar      *label;
	
	g_return_if_fail(gte != NULL);
	g_return_if_fail(GTK_MATRIX(gte->matrix)->maxrow > row);
	
	rec = gtrans_table_editor_find_record(gte,row);
	if (rec != NULL)
		return;
	
	rec = g_new(GTE_Record,1);
	
	rec->row  = row;
	rec->prev = GTE_UPDATE;
	rec->type = GTE_UPDATE;
	
	gte->mod = g_list_append(gte->mod,rec);
	
	label = g_strdup("*");
	gtk_matrix_row_button_add_label(GTK_MATRIX(gte->matrix),row,label);
	g_free(label);
}






/**
 * gtrans_table_editor_entry_changed
 */
static void
gtrans_table_editor_entry_changed(GtkMatrix         *matrix,
				  gint               row,
				  gint               col,
				  GTransTableEditor *gte)
{
      	if (matrix->maxrow != row)
		gtrans_table_editor_toggle_row_mod(gte,row);
	else
		gtrans_table_editor_toggle_row_new(gte,row);
}




static void
gtrans_table_editor_popup_row_menu(GtkMatrix         *matrix,
				   gint               row,
				   gint               col,
				   GTransTableEditor *gte)
{
	GdkEventButton    event;
	gint              index;
	
	index = gnome_popup_menu_do_popup_modal(gte->row_menu,NULL,NULL,&event,NULL);
	
	switch (index){
	case 1:
		gtrans_table_editor_toggle_row_del(gte,row);
		break;
	case 2:
		gtrans_table_editor_toggle_row_undel(gte,row);
		break;
	}
}






/**
 *-----------------------------------------------------------------------------
 * Building the table editor
 *-----------------------------------------------------------------------------
 */


/**
 * gtrans_table_editor_build_matrix:
 * @gte  : The GTransTableEditor
 * 
 * Builds the GtkMatrix for the fields name and type 
 * definition.
 * This funcion must be called after @gte->field_types
 * has been retrived correctly from the plugin set.
 */
static void
gtrans_table_editor_build_matrix(GTransTableEditor *gte)
{
	GTransIFaseResult  *res;
	gchar             **tmp;
	gint                j = 0, i, rows, cols;   
	
	
	res  = gte->res;
	rows = res->n_rows;
	cols = res->n_cols;
	
	gte->matrix = gtk_matrix_new(rows+1,cols);
	
	GTK_MATRIX_SET_FLAGS(GTK_MATRIX(gte->matrix),GTK_MATRIX_AUTORESIZE);
	GTK_MATRIX_SET_FLAGS(GTK_MATRIX(gte->matrix),GTK_MATRIX_ROW_FROZEN);	
	
	gtk_signal_connect(GTK_OBJECT(gte->matrix),"entry_changed",
			   GTK_SIGNAL_FUNC(gtrans_table_editor_entry_changed),
			   (gpointer)gte);

	gtk_signal_connect(GTK_OBJECT(gte->matrix),"row_button_press",
			   GTK_SIGNAL_FUNC(gtrans_table_editor_popup_row_menu),
			   (gpointer)gte);
	
	
	for (i = 0 ; i < cols ; i++)
		gtk_matrix_column_button_add_label(GTK_MATRIX(gte->matrix),i,res->fields[i]);
	
	while ((tmp = gtrans_ifase_result_fetch_row(res)) != NULL){
		for (i = 0 ; i < cols ; i++)
			gtk_matrix_set_cell_text(GTK_MATRIX(gte->matrix),j,i,tmp[i]);
		j++;
	}
	gtrans_ifase_result_rewind(res);
}






/**
 * gtrans_table_editor_create_row_popup:
 * @gte : The GTransTableEditor
 *
 * Creates the popup menu for the row buttons.
 */
static void
gtrans_table_editor_build_row_popup(GTransTableEditor *gte)
{
	GnomeUIInfo row_popup_menu[] = {
		GNOMEUIINFO_SEPARATOR,
		{ 
			GNOME_APP_UI_ITEM,
			N_("Delete this register"),
			N_("Tag this register for deletion"),
			NULL,
			NULL,
			NULL,
			GNOME_APP_PIXMAP_STOCK,
			GNOME_STOCK_MENU_TRASH,
			0,
			GDK_CONTROL_MASK,
			NULL
		},
		{ 
			GNOME_APP_UI_ITEM,
			N_("Undelete this register"),
			N_("Untag this register for deletion"),
			NULL,
			NULL,
			NULL,
			GNOME_APP_PIXMAP_STOCK,
			GNOME_STOCK_MENU_UNDELETE,
			0,
			GDK_CONTROL_MASK,
			NULL
		},
		GNOMEUIINFO_END
	};
	
	gte->row_menu = gnome_popup_menu_new(row_popup_menu);
}





/*
 * gtrans_table_editor_new
 * @conn  : The GTransIFaseConn to the database we are working on.
 * @res   : The GtransIFaseResult of the "SELECT *" query.
 * @owner : The widget who will be the owner of the editor,
 *          should be the GTransDbEditor who poped it.
 *
 * Create a new GTransTableEditor which will commit the design
 * to @conn and signal to @owner that should update itself.
 * A GTransTableEditor allows us to create new tables in the database.
 */
GtkWidget *
gtrans_table_editor_new(GTransIFaseConn   *conn,
			GTransIFaseResult *res,
			gchar             *tname,
			GtkWidget         *owner)
{
	GTransTableEditor     *gte;
	GtkWidget             *vbox;
	
	GtkWidget             *hbox;
	GtkWidget             *label1;
	gchar                 *str;
	
	GtkWidget             *sep1;
	GtkWidget             *scrolled;
	
	GtkWidget             *hbutt_box;
	GtkWidget             *all_button;
	GtkWidget             *next_button;
	GtkWidget             *stop_button;
	GtkWidget             *close_button;
	GtkWidget             *sep2;
	
	
	gte = gtk_type_new(gtrans_table_editor_get_type());
	gte->conn  = conn;
	gte->owner = owner;
	gte->res   = res;
	gte->tname = g_strdup(tname);
	
	gtk_window_set_title(GTK_WINDOW(gte),"Table Editor");
	gtk_container_set_border_width(GTK_CONTAINER(gte),10);
	
	vbox = gtk_vbox_new(FALSE,10);
	gtk_container_add(GTK_CONTAINER(gte),vbox);
	
	
	hbox = gtk_hbox_new(FALSE,5);
        gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
	
	str = g_strconcat("Table: ",tname,", Database: ",conn->db,NULL);
	
	label1 = gtk_label_new(str);
	g_free(str);
	gtk_box_pack_start(GTK_BOX(hbox),label1,FALSE,FALSE,0);
	
	
	sep1 = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox),sep1,FALSE,FALSE,0);
	
	
	scrolled = gtk_scrolled_window_new(NULL,NULL);
	gtk_widget_set_usize(scrolled,450,350);
	gtk_box_pack_start_defaults(GTK_BOX(vbox),scrolled);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
	
	/* Create the matrix */
	gtrans_table_editor_build_matrix(gte);
	gtk_container_add(GTK_CONTAINER(scrolled),gte->matrix);
	
	gtrans_table_editor_build_row_popup(gte);
	
	sep2 = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox),sep2,FALSE,FALSE,0);
	
	hbutt_box = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbutt_box),GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbutt_box),0);
        gtk_box_pack_start(GTK_BOX(vbox),hbutt_box,FALSE,FALSE,0);
	
	next_button = gtrans_button_with_stock_n_label(GNOME_STOCK_PIXMAP_FORWARD,"Next");
	GTK_WIDGET_SET_FLAGS(next_button,GTK_CAN_DEFAULT);
	gtk_box_pack_end(GTK_BOX(hbutt_box),next_button,TRUE,TRUE,0);
	gtk_signal_connect_object(GTK_OBJECT(next_button),"clicked",
				  GTK_SIGNAL_FUNC(gtrans_table_editor_emit__commit_next),
				  GTK_OBJECT(gte));

	all_button = gtrans_button_with_stock_n_label(GNOME_STOCK_PIXMAP_LAST,"All");
	GTK_WIDGET_SET_FLAGS(all_button,GTK_CAN_DEFAULT);
	gtk_box_pack_end(GTK_BOX(hbutt_box),all_button,TRUE,TRUE,0);
	gtk_signal_connect_object(GTK_OBJECT(all_button),"clicked",
				  GTK_SIGNAL_FUNC(gtrans_table_editor_emit__commit_all),
				  GTK_OBJECT(gte));
	
	stop_button = gtrans_button_with_stock_n_label(GNOME_STOCK_PIXMAP_STOP,"Stop");
	GTK_WIDGET_SET_FLAGS(stop_button,GTK_CAN_DEFAULT);
	gtk_box_pack_end(GTK_BOX(hbutt_box),stop_button,TRUE,TRUE,0);
	gtk_signal_connect_object(GTK_OBJECT(stop_button),"clicked",
				  GTK_SIGNAL_FUNC(gtrans_table_editor_emit__stop),
				  GTK_OBJECT(gte));
	
	close_button = gnome_stock_button(GNOME_STOCK_BUTTON_CLOSE);
	GTK_WIDGET_SET_FLAGS(close_button,GTK_CAN_DEFAULT);
	gtk_box_pack_end(GTK_BOX(hbutt_box),close_button,TRUE,TRUE,0);
	gtk_widget_grab_default(close_button);
	gtk_signal_connect_object(GTK_OBJECT(close_button),"clicked",
				  GTK_SIGNAL_FUNC(gtrans_table_editor_emit__close),
				  GTK_OBJECT(gte));
	
	gtk_signal_connect_object(GTK_OBJECT(gte),"delete-event",
				  GTK_SIGNAL_FUNC(gtrans_table_editor_emit__close),
				  GTK_OBJECT(gte));
	
	
	gtk_widget_show_all(vbox);
	
	return GTK_WIDGET(gte); 
}






/**
 *-----------------------------------------------------------------------------
 * Reading data from the editor
 *-----------------------------------------------------------------------------
 */

/**
 * gtrans_table_editor_get_new_data_arry
 * @gte   : The GTransTableEditor
 * @row   : The number of a row
 *
 * Return the arry of strings of the new data of the row
 * indexed @row in the @gte.
 * The "new data" is the data currently at sight in the 
 * @gte.
 */
static gchar **
gtrans_table_editor_get_new_data_arry(GTransTableEditor *gte,
				      gint               row)
{
	GtkMatrix  *matrix;
	gchar     **data, *text;
	gint        cols ,i;
	
	g_return_val_if_fail(gte != NULL,NULL);
	g_return_val_if_fail(gte->matrix != NULL,NULL);
	g_return_val_if_fail(GTK_MATRIX(gte->matrix)->maxrow > row,NULL);
	
	matrix = GTK_MATRIX(gte->matrix);
	cols = matrix->maxcol + 1;
	
	data = g_new0(gchar*,cols);
	
	for ( i = 0 ; i < cols ; i++){
		text = gtk_matrix_cell_get_text(matrix,row,i);
		data[i] = g_strdup(text);
	}
	
	return data;
}




/**
 * gtrans_table_editor_old_res_data_arry
 * @gte   : The GTransTableEditor
 * @row   : The number of a row
 *
 * Return the arry of strings of the old data of
 * the row indexed @row, in the GTransIFaseResult
 * that we passed when we bulded the GTransTablEditor.
 * The "old data" is the initial data (The GTransIFaseResult) 
 * that we entred in the @gte.
 */
static gchar **
gtrans_table_editor_get_old_data_arry(GTransTableEditor *gte,
				      gint               row)
{
	GTransIFaseResult  *res;
	gchar             **tmp, **data;
	gint                cols ,i;
	
	g_return_val_if_fail(gte != NULL,NULL);
	g_return_val_if_fail(gte->matrix != NULL,NULL);
	g_return_val_if_fail(gte->res->n_rows >= row,NULL);
	
	res  = gte->res;
	cols = res->n_cols;
	
	data = g_new0(gchar*,cols);
	
	tmp = gtrans_ifase_result_get_row(res,row);
	
	for ( i = 0 ; i < cols ; i++)
		data[i] = g_strdup(tmp[i]);
	
	return data;
}






/**
 *-----------------------------------------------------------------------------
 * Query creation funcs
 *-----------------------------------------------------------------------------
 */


/**
 * gtrans_table_editor_create_delete_query
 * @tname    : The name of the table to be updated
 * @fields   : The (NULL terminated) arry of string containing
 *             the name of the fields of this table.
 * @new      : The arry of string of the new values.
 *
 * Returns   : The query to commit the new values to the server.
 */
static gchar *
gtrans_table_editor_create_delete_query(gchar   *tname,
					gchar  **fields,
					gchar  **old)
{
	gchar        *qstr;
	gint          i, size;
	
	/* alloc mem */
	size = 20 + strlen(tname);
	
	for (i = 0 ; fields[i] != NULL ; i++)
		if (old[i] == NULL)
			size += strlen(fields[i]) + 8 + 5;
		else
			size += strlen(fields[i]) + 2 + 
				gtrans_string_escaped_len(old[i]) + 6;
	
	qstr = g_new0(gchar,size);
	
	strcpy(qstr,"DELETE FROM ");
	strcat(qstr,tname);
	strcat(qstr," WHERE ");
	
	for( i = 0 ; fields[i] != NULL ; i++){
		
		strcat(qstr,fields[i]);
		
		if (old[i] == NULL)
			strcat(qstr," IS NULL");
		else {
			strcat(qstr,"='");
			gtrans_string_escaped_concat(qstr,old[i]);
			strcat(qstr,"'");
		}
		
		if(fields[i+1] != NULL)
			strcat(qstr," AND ");
	}
	
	return qstr;
}




/**
 * gtrans_table_editor_create_insert_query
 * @tname    : The name of the table to be updated
 * @fields   : The (NULL terminated) arry of string containing
 *             the name of the fields of this table.
 * @new      : The arry of string of the new values.
 *
 * Returns   : The query to commit the new values to the server.
 */
static gchar *
gtrans_table_editor_create_insert_query(gchar   *tname,
					gchar  **fields,
					gchar  **new)
{
	gchar        *qstr;
	gint          i, size;
	
	/* alloc mem */
	size = 25 + strlen(tname);
	
	for (i = 0 ; fields[i] != NULL ; i++)
		if (new[i] == NULL)
			size += 5;
		else
			size += gtrans_string_escaped_len(new[i]) + 3;
	
	
	qstr = g_new0(gchar,size);
	
	strcpy(qstr,"INSERT INTO ");
	strcat(qstr,tname);
	strcat(qstr," VALUES ( ");
	
	for( i = 0 ; fields[i] != NULL ; i++){
		if (new[i] == NULL)
			strcat(qstr,"NULL");
		else {
			strcat(qstr,"'");
			gtrans_string_escaped_concat(qstr,new[i]);
			strcat(qstr,"'");
		}
		if(fields[i+1] != NULL)
			strcat(qstr,",");
	}
	strcat(qstr," )");
	
	return qstr;
}




/**
 * gtrans_table_editor_create_update_query
 * @tname    : The name of the table to be updated
 * @fields   : The (NULL terminated) arry of string containing
 *             the name of the fields of this table.
 * @old      : The arry of string of the old values.
 * @new      : The arry of string of the new values.
 *
 * Returns   : The query to commit this changes to the server.
 */
static gchar *
gtrans_table_editor_create_update_query(gchar   *tname,
					gchar  **fields,
					gchar  **old,
					gchar  **new)
{
	gchar        *qstr;
	gint          i, size, sizeold, sizenew;
	
	/* Calc the size of the query */
	size = 20 + strlen(tname);
	for (i = 0 ; fields[i] != NULL ; i++){
		
		if (new[i] == NULL )
			sizenew = 4;
		else
			sizenew = gtrans_string_escaped_len(new[i]) + 2;
		
		if (old[i] == NULL )
			sizeold = 7;
		else
			sizeold = gtrans_string_escaped_len(old[i]) + 2;
		
		size += sizenew + sizeold + (2*strlen(fields[i])) + 8;
	}
	
	qstr = g_new0(gchar,size);
	
	strcpy(qstr,"UPDATE ");
	strcat(qstr,tname);
	strcat(qstr," SET ");
	
	for( i = 0 ; fields[i] != NULL ; i++){
		strcat(qstr,fields[i]);
		strcat(qstr,"=");
		
		if (new[i] == NULL)
			strcat(qstr,"NULL");
		else {
			strcat(qstr,"'");
			gtrans_string_escaped_concat(qstr,new[i]);
			strcat(qstr,"'");
		}
		if(fields[i+1] != NULL)
			strcat(qstr,",");
	}
	
	strcat(qstr," WHERE ");
	for( i = 0 ; fields[i] != NULL ; i++){
		strcat(qstr,fields[i]);
		
		if (old[i] == NULL)
			strcat(qstr," IS NULL");
		else {
			strcat(qstr,"='");
			gtrans_string_escaped_concat(qstr,old[i]);
			strcat(qstr,"'");
		}
		
		strcat(qstr," ");
		if(fields[i+1] != NULL)
			strcat(qstr,"AND ");
	}
	
     	return qstr;
}







/**
 *-----------------------------------------------------------------------------
 * Query retrival funcs
 *-----------------------------------------------------------------------------
 */



/**
 * gtrans_table_editor_pop_update_query:
 * @gte : The GTransTableEditor
 *
 * Return the query that will commit to the server the changes
 * made to the row standing next in the modified rows list.
 * Once this changes have been commited we can flush the index
 * of the row from the modified rows list and pop the next query.
 */
static gchar *
gtrans_table_editor_pop_update_query(GTransTableEditor *gte)
{
	GTE_Record *mod;
	gchar      *qstr, *tname, **fields, **old, **new;
	gint        row;
	
	g_return_val_if_fail(gte != NULL, NULL);
	g_return_val_if_fail(gte->mod != NULL, NULL);

	mod = (GTE_Record*)gte->mod->data;
	row = mod->row;
	
	fields = gte->res->fields;
	tname  = gte->tname;
	
	old = gtrans_table_editor_get_old_data_arry(gte,row);
	new = gtrans_table_editor_get_new_data_arry(gte,row);
	
	qstr = gtrans_table_editor_create_update_query(tname,fields,old,new);
	
	g_free (old);
	g_free (new);

	return qstr;
}





/**
 * gtrans_table_editor_pop_insert_query:
 * @gte : The GTransTableEditor
 *
 * Return the query that will commit to the server the new
 * values entered in the row standing next in the new rows list.
 * Once this changes have been commited we can flush the index
 * of the row from the new rows lists and pop the next query.
 */
static gchar *
gtrans_table_editor_pop_insert_query(GTransTableEditor *gte)
{
	GTE_Record *mod;
	gchar      *qstr, *tname, **new, **fields;
	gint        row;
	
	g_return_val_if_fail(gte != NULL, NULL);
	g_return_val_if_fail(gte->mod != NULL, NULL);
	
	mod = (GTE_Record*)gte->mod->data;
	row = mod->row;
	
	tname  = gte->tname;
	fields = gte->res->fields;
	
	new = gtrans_table_editor_get_new_data_arry(gte,row);
	
	qstr = gtrans_table_editor_create_insert_query(tname,fields,new);
	
	g_free (new);
	
	return qstr;
}




/**
 * gtrans_table_editor_pop_delete_query:
 * @gte : The GTransTableEditor
 *
 * Return the query that will commit to the server the new
 * values entered in the row standing next in the new rows list.
 * Once this changes have been commited we can flush the index
 * of the row from the new rows lists and pop the next query.
 *
 * Returns NULL on one exeption : If the row to be deleted
 * in new in @gte and has not yet been commited.
 */
static gchar *
gtrans_table_editor_pop_delete_query(GTransTableEditor *gte)
{
	GTE_Record *rec;
	gchar      *qstr, *tname, **old, **fields;
	gint        row;
	
	g_return_val_if_fail(gte != NULL, NULL);
	g_return_val_if_fail(gte->mod != NULL, NULL);
	
	rec = (GTE_Record*)gte->mod->data;
	
	if (rec->prev == GTE_INSERT)
		return NULL;
	
	row = rec->row;
	
	tname  = gte->tname;
	fields = gte->res->fields;
	
	old = gtrans_table_editor_get_old_data_arry(gte,row);
	
	qstr = gtrans_table_editor_create_delete_query(tname,fields,old);
	
	g_free (old);
	
	return qstr;
}




/**
 * gtrans_table_editor_pop_query:
 * @gte : The GTransTableEditor
 *
 * Return the query that will commit to the server the changes
 * made to the row standing next in the mod/new lists.
 * Once this changes have been commited we can flush the index
 * of the row from the mod/new lists and pop the next query.
 */
gchar *
gtrans_table_editor_pop_query(GTransTableEditor *gte)
{
	GTE_Record *rec;
	
	if (gte->mod == NULL)
		return NULL;
	
	rec = (GTE_Record*) gte->mod->data;
	
	if (rec->type == GTE_DELETE)
		return gtrans_table_editor_pop_delete_query(gte);
	
	if (rec->type == GTE_UPDATE)
		return gtrans_table_editor_pop_update_query(gte);
	
	if (rec->type == GTE_INSERT)
		return gtrans_table_editor_pop_insert_query(gte);
	
	return NULL;
}






/**
 *-----------------------------------------------------------------------------
 * Syncronizing changes made
 *-----------------------------------------------------------------------------
 */


/**
 * gtrans_table_editor_sync_insert:
 * @gte : The GTransTableEditor
 * @rec : The GTE_Record of the change made
 *
 * Syncronizes the 'insert' change made in the Matrix to the
 * GTransIFaseResult statnding in the back. This is,
 * insert the new row to the GTransIFaseResult.
 */
static void
gtrans_table_editor_sync_insert(GTransTableEditor *gte,
				GTE_Record        *rec)
{
	GTransIFaseResult  *res;
	gchar             **data;
	gint                row;

	row = rec->row;
	
	data = gtrans_table_editor_get_new_data_arry(gte,row);
	if (data == NULL)
		return;
	
	res = gte->res; 
	gtrans_ifase_result_append_row(res,data);
	gtk_matrix_row_button_add_label(GTK_MATRIX(gte->matrix),row,NULL);
}




/**
 * gtrans_table_editor_sync_update:
 * @gte : The GTransTableEditor
 * @rec : The GTE_Record of the change made
 *
 * Syncronizes the 'update' change made in the Matrix to the
 * GTransIFaseResult statnding in the back. This is,
 * update the changes made in the GTransIFaseResult row.
 */
static void
gtrans_table_editor_sync_update(GTransTableEditor *gte,
				GTE_Record        *rec)
{
	GTransIFaseResult  *res;
	gchar             **data;
	gint                row;
	
	row = rec->row;

	data = gtrans_table_editor_get_new_data_arry(gte,row);
	if (data == NULL)
		return;
	
	res = gte->res;
	gtrans_ifase_result_replace_row(res,row,data);

	gtk_matrix_row_button_add_label(GTK_MATRIX(gte->matrix),row,NULL);
}




/**
 * gtrans_table_editor_sync_delete:
 * @gte : The GTransTableEditor
 * @rec : The GTE_Record of the change made
 *
 * Syncronizes the 'delete' change made in the Matrix to the
 * GTransIFaseResult statnding in the back. This is,
 * remove the row from the GTransIFaseResult.
 */
static void
gtrans_table_editor_sync_delete(GTransTableEditor *gte,
				GTE_Record        *rec)
{
	GTransIFaseResult  *res;
	gint                row;
	GList              *tmp_list;
	GTE_Record         *tmp_rec;
	

	row = rec->row;
	
	/* If it has just been added to the editor, we dont need to
	 * remove it from the result */
	if (rec->prev != GTE_INSERT){
		res = gte->res;
		gtrans_ifase_result_remove_row(res,row);
	}
	
	gtk_matrix_delete_rows(GTK_MATRIX(gte->matrix),row,1);
	

	/* Since we removed a row from the Matrix, we need to update
	 * all the record's row numbers past that row */
	for (tmp_list = gte->mod->next ; 
	     tmp_list != NULL ; 
	     tmp_list = tmp_list->next){
		
		tmp_rec = (GTE_Record *)tmp_list->data;
		if (tmp_rec->row > row)
			tmp_rec->row--;
	}
}





/**
 * gtrans_table_editor_sync:
 * @gte  : The GTransTableEditor
 *
 * Syncronizes the matrix widget and the GTransIFaseResult standing in the back,
 * this is, updates the changes made to the matrix widget.
 * Also removes from @gte->mod the record of the modification
 * standing in line.
 * @gte->mod is a GList * of all the modification records appended.
 * The records are GTE_Record structs.
 */
gint
gtrans_table_editor_sync(GTransTableEditor *gte)
{
	GTE_Record *rec;
	
	if (gte->mod != NULL){
		
		rec = (GTE_Record*)gte->mod->data;
		
		switch (rec->type){
			
		case GTE_UPDATE:
			gtrans_table_editor_sync_update(gte,rec);
			break;
			
		case GTE_INSERT:
			gtrans_table_editor_sync_insert(gte,rec);
			break;

		case GTE_DELETE:
			gtrans_table_editor_sync_delete(gte,rec);
			break;
		}
		
		g_free(rec);
		gte->mod = g_list_remove_link(gte->mod,gte->mod);
		return 0;
	}
	return 1;
}







/**
 *-----------------------------------------------------------------------------
 * Sequence funcs. 
 *-----------------------------------------------------------------------------
 */


/**
 * gtrans_table_editor_next_row:
 * @gte : The GTransTableEditor
 *
 * Return the number of the row standing next in the
 * modification list (@gte->mod).
 * Return -1 if there is not any.
 */
gint 
gtrans_table_editor_next_row(GTransTableEditor *gte)
{
	GTE_Record *rec;
	
	if (gte->mod == NULL)
		return -1;
	
	rec = (GTE_Record*)gte->mod->data;
	
	return rec->row;
}





/**
 * gtrans_table_editor_can_next:
 * @gte : The GTransTableEditor
 *
 * Return 1 if there are changes to be done,
 * return 0 otherwise.
 */
gint 
gtrans_table_editor_can_next(GTransTableEditor *gte)
{
	if (gte->mod == NULL)
		return 0;
	
	return 1;
}





/**
 * gtrans_table_editor_inetrrupt:
 * @gte : The GTransTableEditor
 *
 * Return 1 and unset the interrupt flag if it is already set.
 * Return 0 otherwise.
 */
gint
gtrans_table_editor_interrupt(GTransTableEditor *gte)
{
	if (!GTRANS_TABLE_EDITOR_INTERRUPT(gte))
		return 0;
	
	GTRANS_TABLE_EDITOR_UNSET_FLAGS(gte,GTRANS_TABLE_EDITOR_INTERRUPT);
	return 1;
}
