/*
 * GooCanvas. Copyright (C) 2005-6 Damon Chaplin.
 * Released under the GNU LGPL license. See COPYING for details.
 *
 * goocanvaspath.h - a path item, very similar to the SVG path element.
 */
#ifndef __GOO_CANVAS_PATH_H__
#define __GOO_CANVAS_PATH_H__

#include <gtk/gtk.h>
#include "goocanvasitemsimple.h"

G_BEGIN_DECLS


#define GOO_TYPE_CANVAS_PATH            (goo_canvas_path_get_type ())
#define GOO_CANVAS_PATH(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), GOO_TYPE_CANVAS_PATH, GooCanvasPath))
#define GOO_CANVAS_PATH_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass), GOO_TYPE_CANVAS_PATH, GooCanvasPathClass))
#define GOO_IS_CANVAS_PATH(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GOO_TYPE_CANVAS_PATH))
#define GOO_IS_CANVAS_PATH_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GOO_TYPE_CANVAS_PATH))
#define GOO_CANVAS_PATH_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS ((obj), GOO_TYPE_CANVAS_PATH, GooCanvasPathClass))


/**
 * GooCanvasPathCommandType
 * @GOO_CANVAS_PATH_MOVE_TO: move to the given point.
 * @GOO_CANVAS_PATH_CLOSE_PATH: close the current path, drawing a line from the
 *  current position to the start of the path.
 * @GOO_CANVAS_PATH_LINE_TO: draw a line to the given point.
 * @GOO_CANVAS_PATH_HORIZONTAL_LINE_TO: draw a horizontal line to the given
 *  x coordinate.
 * @GOO_CANVAS_PATH_VERTICAL_LINE_TO: draw a vertical line to the given y
 *  coordinate.
 * @GOO_CANVAS_PATH_CURVE_TO: draw a bezier curve using two control
 *  points to the given point.
 * @GOO_CANVAS_PATH_SMOOTH_CURVE_TO: draw a bezier curve using a reflection
 *  of the last control point of the last curve as the first control point,
 *  and one new control point, to the given point.
 * @GOO_CANVAS_PATH_QUADRATIC_CURVE_TO: draw a quadratic bezier curve using
 *  a single control point to the given point.
 * @GOO_CANVAS_PATH_SMOOTH_QUADRATIC_CURVE_TO: draw a quadratic bezier curve
 *  using a reflection of the control point from the previous curve as the
 *  control point, to the given point.
 * @GOO_CANVAS_PATH_ELLIPTICAL_ARC: draw an elliptical arc, using the given
 *  2 radii, the x axis rotation, and the 2 flags to disambiguate the arc,
 *  to the given point.
 *
 * GooCanvasPathCommandType specifies the type of each command in the path.
 * See the path element in the <ulink url="http://www.w3.org/Graphics/SVG/">
 * Scalable Vector Graphics (SVG) specification</ulink> for more details.
 */
typedef enum
{
  /* Simple commands like moveto and lineto: MmZzLlHhVv. */
  GOO_CANVAS_PATH_MOVE_TO,
  GOO_CANVAS_PATH_CLOSE_PATH,
  GOO_CANVAS_PATH_LINE_TO,
  GOO_CANVAS_PATH_HORIZONTAL_LINE_TO,
  GOO_CANVAS_PATH_VERTICAL_LINE_TO,

  /* Bezier curve commands: CcSsQqTt. */
  GOO_CANVAS_PATH_CURVE_TO,
  GOO_CANVAS_PATH_SMOOTH_CURVE_TO,
  GOO_CANVAS_PATH_QUADRATIC_CURVE_TO,
  GOO_CANVAS_PATH_SMOOTH_QUADRATIC_CURVE_TO,

  /* The elliptical arc commands: Aa. */
  GOO_CANVAS_PATH_ELLIPTICAL_ARC
} GooCanvasPathCommandType;


typedef union _GooCanvasPathCommand  GooCanvasPathCommand;

/* Note that the command type is always the first element in each struct, so
   we can always use it whatever type of command it is. */

/**
 * GooCanvasPathCommand
 *
 * GooCanvasPathCommand holds the data for each command in the path.
 *
 * The @relative flag specifies that the coordinates for the command are
 * relative to the current point. Otherwise they are assumed to be absolute
 * coordinates.
 */
union _GooCanvasPathCommand
{
  /* Simple commands like moveto and lineto: MmZzLlHhVv. */
  struct {
    guint type : 5; /* GooCanvasPathCommandType */
    guint relative : 1;
    gdouble x, y;
  } simple;

  /* Bezier curve commands: CcSsQqTt. */
  struct {
    guint type : 5; /* GooCanvasPathCommandType */
    guint relative : 1;
    gdouble x, y, x1, y1, x2, y2;
  } curve;

  /* The elliptical arc commands: Aa. */
  struct {
    guint type : 5; /* GooCanvasPathCommandType */
    guint relative : 1;
    guint large_arc_flag : 1;
    guint sweep_flag : 1;
    gdouble rx, ry, x_axis_rotation, x, y;
  } arc;
};


typedef struct _GooCanvasPath       GooCanvasPath;
typedef struct _GooCanvasPathClass  GooCanvasPathClass;

/**
 * GooCanvasPath
 * @commands: an array of #GooCanvasPathCommand holding the specification of
 *  the path. Applications can modify this directly, but must call
 *  goo_canvas_item_simple_emit_changed() afterwards to notify the
 *  views that the #GooCanvasPath has changed.
 *
 * The #GooCanvasPath-struct contains the list of commands specifying the path.
 */
struct _GooCanvasPath
{
  GooCanvasItemSimple parent;

  /*< public >*/
  GArray *commands;
};

struct _GooCanvasPathClass
{
  GooCanvasItemSimpleClass parent_class;
};


GType           goo_canvas_path_get_type          (void) G_GNUC_CONST;
GooCanvasItem*  goo_canvas_path_new               (GooCanvasItem *parent,
						   gchar         *path_data,
						   ...);


G_END_DECLS

#endif /* __GOO_CANVAS_PATH_H__ */
