/*
 * GooCanvas. Copyright (C) 2005 Damon Chaplin.
 * Released under the GNU LGPL license. See COPYING for details.
 *
 * goocanvasitemview.h - interface for views of canvas items & groups.
 */
#ifndef __GOO_CANVAS_ITEM_VIEW_H__
#define __GOO_CANVAS_ITEM_VIEW_H__

#include <gtk/gtk.h>
#include "goocanvasitem.h"

G_BEGIN_DECLS


#define GOO_TYPE_CANVAS_ITEM_VIEW            (goo_canvas_item_view_get_type ())
#define GOO_CANVAS_ITEM_VIEW(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), GOO_TYPE_CANVAS_ITEM_VIEW, GooCanvasItemView))
#define GOO_CANVAS_ITEM_VIEW_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass), GOO_TYPE_CANVAS_ITEM_VIEW, GooCanvasItemViewClass))
#define GOO_IS_CANVAS_ITEM_VIEW(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GOO_TYPE_CANVAS_ITEM_VIEW))
#define GOO_CANVAS_ITEM_VIEW_GET_IFACE(obj)  (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GOO_TYPE_CANVAS_ITEM_VIEW, GooCanvasItemViewIface))


/**
 * GooCanvasItemView
 *
 * #GooCanvasItemView is a typedef used for objects that implement the
 * #GooCanvasItemView interface.
 *
 * (There is no actual #GooCanvasItemView struct, since it is only an
 * interface. But using '#GooCanvasItemView' is more helpful than using
 * '#GObject'.)
 */

/**
 * GooCanvasItemViewIface
 *
 * #GooCanvasItemViewIFace holds the virtual methods that make up the
 * #GooCanvasItemView interface.
 */
typedef struct _GooCanvasItemViewIface  GooCanvasItemViewIface;

struct _GooCanvasItemViewIface
{
  /*< private >*/
  GTypeInterface base_iface;

  /*< public >*/
  /* Virtual methods that group views should implement. */
  GooCanvasView*       (* get_canvas_view)  (GooCanvasItemView   *view);
  gint		       (* get_n_children)   (GooCanvasItemView   *group_view);
  GooCanvasItemView*   (* get_child)	    (GooCanvasItemView   *group_view,
					     gint                 child_num);
  void                 (* request_update)   (GooCanvasItemView   *view);

  /* Virtual methods all item views must implement. */
  GooCanvasItemView*   (* get_parent_view)  (GooCanvasItemView   *view);
  void                 (* set_parent_view)  (GooCanvasItemView   *view,
					     GooCanvasItemView   *parent_view);

  cairo_matrix_t*      (* get_transform)    (GooCanvasItemView   *view);
  void                 (* set_transform)    (GooCanvasItemView   *view,
					     cairo_matrix_t      *matrix);

  GooCanvasItem*       (* get_item)	    (GooCanvasItemView   *view);
  void                 (* get_bounds)	    (GooCanvasItemView   *view,
					     GooCanvasBounds     *bounds);
  GooCanvasItemView*   (* get_item_view_at) (GooCanvasItemView   *view,
					     gdouble              x,
					     gdouble              y,
					     cairo_t             *cr,
					     gboolean             is_pointer_event,
					     gboolean             parent_is_visible);
  gboolean             (* is_visible)	    (GooCanvasItemView   *view);
  void                 (* update)           (GooCanvasItemView   *view,
					     gboolean             entire_tree,
					     cairo_t             *cr,
					     GooCanvasBounds     *bounds);
  void		       (* paint)	    (GooCanvasItemView   *view,
					     cairo_t             *cr,
					     GooCanvasBounds     *bounds,
					     gdouble              scale);

  /* Signals. */
  gboolean	       (* enter_notify_event)	(GooCanvasItemView   *view,
						 GooCanvasItemView   *target,
						 GdkEventCrossing    *event);
  gboolean	       (* leave_notify_event)	(GooCanvasItemView   *view,
						 GooCanvasItemView   *target,
						 GdkEventCrossing    *event);
  gboolean	       (* motion_notify_event)	(GooCanvasItemView   *view,
						 GooCanvasItemView   *target,
						 GdkEventMotion      *event);
  gboolean	       (* button_press_event)	(GooCanvasItemView   *view,
						 GooCanvasItemView   *target,
						 GdkEventButton      *event);
  gboolean	       (* button_release_event)	(GooCanvasItemView   *view,
						 GooCanvasItemView   *target,
						 GdkEventButton      *event);
  gboolean	       (* focus_in_event)	(GooCanvasItemView   *view,
						 GooCanvasItemView   *target,
						 GdkEventFocus       *event);
  gboolean	       (* focus_out_event)	(GooCanvasItemView   *view,
						 GooCanvasItemView   *target,
						 GdkEventFocus       *event);
  gboolean	       (* key_press_event)	(GooCanvasItemView   *view,
						 GooCanvasItemView   *target,
						 GdkEventKey         *event);
  gboolean	       (* key_release_event)	(GooCanvasItemView   *view,
						 GooCanvasItemView   *target,
						 GdkEventKey         *event);
  gboolean	       (* grab_broken_event)	(GooCanvasItemView   *view,
						 GooCanvasItemView   *target,
						 GdkEventGrabBroken  *event);
};


GType              goo_canvas_item_view_get_type (void) G_GNUC_CONST;


/*
 * Group view functions - these should only be called on container views.
 */
gint               goo_canvas_item_view_get_n_children  (GooCanvasItemView *view);
GooCanvasItemView* goo_canvas_item_view_get_child       (GooCanvasItemView *view,
							 gint               child_num);
gint               goo_canvas_item_view_find_child      (GooCanvasItemView *view,
							 GooCanvasItemView *child);


/*
 * Item view functions - these are safe to call on all item views.
 */
GooCanvasView*     goo_canvas_item_view_get_canvas_view (GooCanvasItemView *view);
GooCanvasItemView* goo_canvas_item_view_get_parent_view (GooCanvasItemView  *view);
void               goo_canvas_item_view_set_parent_view (GooCanvasItemView  *view,
							 GooCanvasItemView  *parent_view);
gboolean           goo_canvas_item_view_is_container    (GooCanvasItemView *view);

cairo_matrix_t*    goo_canvas_item_view_get_transform   (GooCanvasItemView *view);
void               goo_canvas_item_view_set_transform   (GooCanvasItemView *view,
							 cairo_matrix_t    *matrix);
gboolean           goo_canvas_item_view_get_combined_transform (GooCanvasItemView *view,
								cairo_matrix_t    *result);

GooCanvasItem*     goo_canvas_item_view_get_item    (GooCanvasItemView  *view);
void               goo_canvas_item_view_get_bounds  (GooCanvasItemView  *view,
						     GooCanvasBounds    *bounds);
GooCanvasItemView* goo_canvas_item_view_get_item_view_at (GooCanvasItemView  *view,
							  gdouble             x,
							  gdouble             y,
							  cairo_t            *cr,
							  gboolean            is_pointer_event,
							  gboolean            parent_is_visible);
gboolean           goo_canvas_item_view_is_visible  (GooCanvasItemView   *view);
void               goo_canvas_item_view_request_update (GooCanvasItemView *view);
void		   goo_canvas_item_view_ensure_updated (GooCanvasItemView *view);
void               goo_canvas_item_view_update      (GooCanvasItemView  *view,
						     gboolean            entire_tree,
						     cairo_t            *cr,
						     GooCanvasBounds    *bounds);
void               goo_canvas_item_view_paint       (GooCanvasItemView  *view,
						     cairo_t            *cr,
						     GooCanvasBounds    *bounds,
						     gdouble             scale);


G_END_DECLS

#endif /* __GOO_CANVAS_ITEM_VIEW_H__ */
