/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  satplot.c: Radial plot showing a satellite pass.

  Copyright (C)  2001-2003  Ran Hadary & Alexandru Csete.

  Authors:   Ran Hadary <hadary@users.sourceforge.net>
             Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/


#include <gnome.h>
#include <gconf/gconf-client.h>
#include <math.h>
#include <libgnomeui/gnome-window-icon.h>
#include "defaults.h"
#include "satdata.h"
#include "qth.h"
#include "util.h"
#include "satlog.h"
#include "extra-widgets.h"
#include "satplot-print.h"
#include "satplot.h"



#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif

#define PSIZE_X 500
#define PSIZE_Y 500
#define CENTER_X 250
#define CENTER_Y 250
#define RADIUS 240



extern GtkWidget *app; /* Defined in Main */
extern GConfClient *client;
extern qth_struc qth;

/* Internal Variables  */
GtkWidget *dialog;
GnomeCanvas *canvas;

extern GConfClient *client; /* main.c */

/* Private Functions prototype                                     */
xypoint_t ConvertAzEl (gdouble, gdouble, field_of_view_t);
static void PlotText (xypoint_t,  gchar *, gchar *);
static void PlotTextEast (xypoint_t,  gchar *, gchar *);
gint PlotCircle (gdouble, gchar*, uint);
gdouble ElRadius (gdouble);
void satplot_close_dialog (GtkWidget*, gpointer);
void PlotRadial (xypoint_t,  gchar*);
static void satplot_save_cb            (GtkWidget *, GtkWidget *);
static void satplot_store_filename     (GtkWidget *, GtkWidget *);
static void satplot_filesel_destroy_cb (GtkWidget *, GtkWidget *);
/* End Private functions                                           */


void sat_plot_aos_track (sat_t *sat, gchar **aosdet, guint size)
{

	gchar *buff, **vbuff;
	guint index, x,y;
	GnomeCanvasPoints *track;
	xypoint_t step;
	GtkWidget *gCanvas;
	gdouble radial;

	/* Create the canvas */
	gtk_widget_push_visual (gdk_rgb_get_visual ());
	gtk_widget_push_colormap (gdk_rgb_get_cmap ());
	if (gconf_client_get_bool (client, SATPLOT_NOAACANV_PATH, NULL))
		gCanvas = gnome_canvas_new ();
	else
		gCanvas = gnome_canvas_new_aa ();
	gtk_widget_pop_visual ();
	gtk_widget_pop_colormap ();

	canvas = GNOME_CANVAS (gCanvas);

	/* white background */
	gnome_canvas_item_new (gnome_canvas_root(canvas),
			       gnome_canvas_rect_get_type(),
			       "x1",(double) 0,
			       "y1",(double) 0,
			       "x2",(double) PSIZE_X,
			       "y2",(double) PSIZE_Y,
			       "fill_color", "white",
			       NULL);


	/* Add - Set size and scroll */ 
	/* set where can the canvas scroll (our usable area) */
	gnome_canvas_set_scroll_region (canvas, 0.0, 0.0, PSIZE_X, PSIZE_Y);

	/* set the size of the widget */
	gtk_widget_set_usize (gCanvas, PSIZE_X, PSIZE_Y);

	/* Extra features/texts */
	/* satellite name */
	if (gconf_client_get_bool (client, SATPLOT_SHOWSAT_PATH, NULL)) {
		step.x = 2.0;
		step.y = 5.0;
		buff = g_strconcat (_("SAT: "), sat->name, NULL);
		PlotText (step, _("black"), buff);
		g_free (buff);
	}
	/* AOS/LOS times */
	if (gconf_client_get_bool (client, SATPLOT_SHOWAOS_PATH, NULL)) {
		/* AOS */
		step.x = 2.0;
		step.y = 15.0;
		vbuff = g_strsplit (aosdet[1], ";", 5);
		buff = g_strconcat (_("AOS: "), vbuff[0], NULL);
		PlotText (step, _("black"), buff);
		g_free (buff);
		g_strfreev (vbuff);
		/* LOS */
		step.x = 2.0;
		step.y = 25.0;
		vbuff = g_strsplit (aosdet[size-1], ";", 5);
		buff = g_strconcat (_("LOS: "), vbuff[0], NULL);
		PlotText (step, _("black"), buff);
		g_free (buff);
		g_strfreev (vbuff);
	}
	/* QTH info */
	if (gconf_client_get_bool (client, SATPLOT_SHOWQTH_PATH, NULL)) {
		step.x = PSIZE_X;
		step.y = 5.0;
		buff = g_strdup (qth.name);
		PlotTextEast (step, _("black"), buff);
		g_free (buff);
		step.x = PSIZE_X;
		step.y = 15.0;
		buff = g_strdup (qth.loc);
		PlotTextEast (step, _("black"), buff);
		g_free (buff);
	}
	/* time */
	if (gconf_client_get_bool (client, SATPLOT_SHOWTIME_PATH, NULL)) {
		step.x = 2.0;
		step.y = PSIZE_Y - 10.0;
		buff = g_strconcat (_("Gnome Predict "), VERSION, NULL);
		PlotText (step, _("black"), buff);
		g_free (buff);
		step.x = PSIZE_X;
		step.y = PSIZE_Y - 10.0;
		buff = g_strdup (dnum2fstr (CurrentDaynum (), TFORM_SATLIST));
		PlotTextEast (step, _("black"), buff);
		g_free (buff);
	}
	/* Create the 'bull-eye' circles    */
	PlotCircle (0, _("dark grey"),1);
	PlotCircle (30, _("dark grey"),1);
	PlotCircle (60, _("dark grey"),1);

	for (radial = 0; radial <= 360; radial += 30) {   
		PlotRadial (ConvertAzEl (radial, -2, 0), _("light grey"));
	}


	/* Print North, South, East and West */
	PlotText(ConvertAzEl(90,0,0),_("dark green"),_("E"));
	PlotText(ConvertAzEl(180,0,0),_("dark green"),_("S"));
	PlotText(ConvertAzEl(0,0,0),_("dark green"), _("N"));
	PlotText(ConvertAzEl(270,0,0),_("dark green"), _("W"));

	/* Create Dialog , Title and Icon   */
	buff = g_strconcat (_("Track Plot for: "), sat->name, NULL);
	dialog = gnome_dialog_new (buff, NULL);
	g_free (buff);
  
	buff = g_strconcat (PACKAGE_PIXMAPS_DIR, "/icons/stock_timer.png", NULL);
	gnome_window_icon_set_from_file (GTK_WINDOW (dialog), buff);
	g_free (buff);
  
	/* Setup, connect and disable buttons                           */
	gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (dialog),
						_("Save"),
						GNOME_STOCK_PIXMAP_SAVE_AS);
	gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (dialog),
						_("Print"),
						GNOME_STOCK_PIXMAP_PRINT);
	gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (dialog),
						_("Close"),
						GNOME_STOCK_PIXMAP_CLOSE);
	/* Connect buttons                                              */
  
	gnome_dialog_button_connect (GNOME_DIALOG (dialog), 0,
				     GTK_SIGNAL_FUNC (satplot_save_cb),
				     gCanvas);
	gnome_dialog_button_connect (GNOME_DIALOG (dialog), 1,
				     GTK_SIGNAL_FUNC (satplot_print_cb),
				     gCanvas); 
	gnome_dialog_button_connect (GNOME_DIALOG (dialog), 2,
				     GTK_SIGNAL_FUNC (satplot_close_dialog),
				     dialog);
	/* disable inavailable functions                               */
/*  	gnome_dialog_set_sensitive (GNOME_DIALOG (dialog), 1, FALSE); */

	/* Set defualt button - Close                                  */      
	gnome_dialog_set_default (GNOME_DIALOG (dialog), 2);

	/* Display the dialog */
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), gCanvas,
			    TRUE, TRUE , 0); 
	gtk_widget_show_all (dialog);

	/* Plot position marks */

	track = gnome_canvas_points_new(size - 1); /* Size also inc. the 1st (non-data) line */ 

	for (index = 1; index < size; index++) {
  
		vbuff = g_strsplit(aosdet[index],";",5);
		x = g_strtod (vbuff[1],NULL);
		y = (g_strtod (vbuff[2],NULL) > 0.0) ? g_strtod (vbuff[2],NULL) : 0.0;
		step = ConvertAzEl(x,y,0);
		track->coords[2*(index-1)] = step.x;
		track->coords[(2*(index-1))+1] = step.y;
		g_strfreev(vbuff);

	}
	gnome_canvas_item_new(gnome_canvas_root(canvas),
			      gnome_canvas_line_get_type(),
			      "points", track,
			      "fill_color", _("blue"),
			      "width_pixels", 2,
			      "first_arrowhead", FALSE,
			      "last_arrowhead",  TRUE,
			      "smooth", TRUE,
			      "arrow_shape_a", (double) 10,
			      "arrow_shape_b", (double) 10,
			      "arrow_shape_c", (double) 3,NULL);
	gnome_canvas_points_free(track);
}


xypoint_t ConvertAzEl(gdouble Az, gdouble El, field_of_view_t fov)
{
	/* Convert a given Az,El to the fixed point on the canvas */

	xypoint_t res;
	gdouble x,y;
	gdouble az_rad;
	gdouble radius;
	guint swap;

	/* Check whether user want N/S E/W swapped */
	swap = gconf_client_get_int (client, SATPLOT_SWAP_PATH, NULL);

	/* Compute Elevation circle                */
	radius = ElRadius (El);

	/* Convert the Azimuth to radians          */ 
	az_rad = 2 * M_PI * (Az / 360); 

	switch (swap) {
	case SATPLOT_NWSE:
		az_rad = 2*M_PI - az_rad;
		break;
	case SATPLOT_SENW:
		az_rad = M_PI - az_rad;
		break;
	case SATPLOT_SWNE:
		az_rad = M_PI + az_rad;
		break;
	default:
		break;

	}

	x = CENTER_X + (radius * sin (az_rad));
	y = CENTER_Y - (radius * cos (az_rad));
 
	res.x = x;
	res.y = y;

	return res;
}

static void
PlotText (xypoint_t pos,  gchar *color, gchar *text)
{
	/* Used for putting a colored text at a position on the canvas */

	gnome_canvas_item_new(gnome_canvas_root(canvas),
			      gnome_canvas_text_get_type(),
			      "x", (double) pos.x,
			      "y", (double) pos.y,
			      "text", text,
			      "font", "fixed",
			      "anchor",GTK_ANCHOR_WEST,
			      "fill_color", color,
			      NULL);
			
}

static void
PlotTextEast (xypoint_t pos,  gchar *color, gchar *text)
{
	/* Used for putting a colored text at a position on the canvas
	   Right anchor
	 */

	gnome_canvas_item_new(gnome_canvas_root(canvas),
			      gnome_canvas_text_get_type(),
			      "x", (double) pos.x,
			      "y", (double) pos.y,
			      "text", text,
			      "font", "fixed",
			      "anchor",GTK_ANCHOR_EAST,
			      "fill_color", color,
			      NULL);
			
}

gint PlotCircle (gdouble El, gchar *color, uint width) 
{
	/* Plot the elavation circles                     */
	/* Also print on the number of the El variable         */
	gdouble radius;
	gchar *text;

	radius = ElRadius( El );

	gnome_canvas_item_new(gnome_canvas_root(canvas),
			      gnome_canvas_ellipse_get_type(),
			      "x1", (double) (CENTER_X - radius),
			      "y1", (double) (CENTER_Y - radius),
			      "x2", (double) (CENTER_X + radius),
			      "y2", (double) (CENTER_Y + radius),
			      "width_pixels", width,
			      "outline_color", color,
			      NULL);

	text = g_strdup_printf("%3d",(gint) El);
	gnome_canvas_item_new(gnome_canvas_root(canvas),
			      gnome_canvas_text_get_type(),
			      "x", (double) (CENTER_X + 2),
			      "y", (double) (CENTER_Y + radius - 4),
			      "text", text,
			      "fill_color", "red",
			      "font", "fixed",
			      "anchor",GTK_ANCHOR_WEST,
			      NULL);
	g_free(text);
  
	return 0;
}

gdouble ElRadius( gdouble El)
{
	/* Calculate the Elevation circle radius           */
	/* o Compute the angle in Radians                  */

	return (gdouble) (RADIUS-RADIUS*El/90.0); /* linear */
}

void satplot_close_dialog (GtkWidget *widget, gpointer data)
{
	/* Close the dialog            */
	gnome_dialog_close (GNOME_DIALOG (data));
}
void PlotRadial (xypoint_t pos,  gchar *color)
{
	/* Used for plotting a radial from the circle outwards */
	GnomeCanvasPoints *points;

	points = gnome_canvas_points_new(2);
	points->coords[0] = CENTER_X;
	points->coords[1] = CENTER_Y;
	points->coords[2] = pos.x;
	points->coords[3] = pos.y;

	gnome_canvas_item_new(gnome_canvas_root(canvas),
			      gnome_canvas_line_get_type(),
			      "points", points,
			      "fill_color", color,
			      "width_pixels", 1, NULL);
	
	gnome_canvas_points_free(points);
			
}


static void
satplot_save_cb  (GtkWidget *widget, GtkWidget *canvas)
{
	/* This function is called when the user clicks on the
	   Save button. It pops up a file selector dialog.
	*/
	GdkImlibImage *tmp = 0;
	GtkWidget *filesel,*filter,*hbox,*label;

	/* get image */
	tmp = gdk_imlib_create_image_from_drawable (canvas->window, NULL,
						    0, 0,
						    PSIZE_X, PSIZE_Y);
	/* create fileselector widget and attach image */
	filesel = gtk_file_selection_new (_("Save Plot As..."));
	gtk_object_set_data (GTK_OBJECT (filesel), "image", (gpointer) tmp);
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION(filesel)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC (satplot_store_filename),
			    (gpointer) filesel);
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION(filesel)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC (satplot_filesel_destroy_cb),
			    (gpointer) filesel);
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION(filesel)->cancel_button),
			    "clicked", GTK_SIGNAL_FUNC (satplot_filesel_destroy_cb),
			    (gpointer) filesel);

	/* Create filter widget */
	label = gtk_label_new (_("File type:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	filter = filesel_graphic_filter ();
	gtk_object_set_data (GTK_OBJECT (filesel), "filter", filter);

	hbox = gtk_hbox_new (TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 5);
	gtk_box_pack_start (GTK_BOX (hbox), filter, TRUE, TRUE, 5);
	gtk_container_add (GTK_CONTAINER (GTK_FILE_SELECTION (filesel)->main_vbox),
			   hbox);

	gtk_widget_show_all (filesel);
}


/* in extra-widgets.c */
extern const gchar *gfx_filter_desc[];
extern const gchar *gfx_filter_ext[];

static void
satplot_store_filename (GtkWidget *button, GtkWidget *filesel)
{
	/* This function is called when the user clicks on the
	   OK-button of the file selector dialog.
	*/
	gchar *fname=NULL,*tmp=NULL;
	gint i=0;
	guint filti;
	GtkWidget *menu,*item;

	menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (gtk_object_get_data (GTK_OBJECT (filesel), "filter")));
	item = gtk_menu_get_active (GTK_MENU (menu));
	filti = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (item), "filter"));

	if (filti >= GFX_FILTER_NUM) {
		satlog_log (SAT_LOG_CRITICAL, _("Selected file type out of range (bug)"));
		gnome_app_error (GNOME_APP (app), _("Selected file type out of range (bug)"));
		return;
	}

	if (filti) {
		/* User selected type */
		tmp = gtk_file_selection_get_filename (GTK_FILE_SELECTION (filesel));
		if (tmp) {
			fname = g_strconcat (tmp, ".", gfx_filter_ext[filti], NULL);
			i = gdk_imlib_save_image ((GdkImlibImage *) gtk_object_get_data (GTK_OBJECT (filesel), "image"),
						  fname, 0);
		}
	}
	else {
		/* Automatic file type from extension */
		fname = gtk_file_selection_get_filename (GTK_FILE_SELECTION (filesel));
		if (fname) {
			i = gdk_imlib_save_image((GdkImlibImage *) gtk_object_get_data (GTK_OBJECT (filesel),
											"image"),
						 fname, 0);
		}
	}
/*  	if (tmp) */
/*  		g_free (tmp); */
/*  	if (fname) */
/*  		g_free (fname); */
	if (i)
		satlog_log (SAT_LOG_INFO, _("SATPLOT: gdk_imlib_save() returned non-zero value."));

}


static void
satplot_filesel_destroy_cb (GtkWidget *button, GtkWidget *filesel)
{
	gdk_imlib_destroy_image ((GdkImlibImage *) gtk_object_get_data (GTK_OBJECT (filesel), "image"));
	gtk_widget_destroy (GTK_WIDGET (gtk_object_get_data (GTK_OBJECT (filesel), "filter")));
	gtk_widget_destroy (filesel);
}
