#include <unistd.h>
#include <stdio.h>
#include <string.h>

#include <orbit/orbit.h>
#include <libbonobo.h>

#include <shell/Evolution.h>

#include "Evolution-DataServer-Mail.h"
#include "e-mail-remote-glue-marshal.h"
#include "e-mail-remote-glue-folderinfo.h"
#include "e-mail-remote-glue-messageinfo.h"
#include "e-mail-remote-glue-sessionchange.h"
#include "e-mail-remote-glue-storechange.h"
#include "e-mail-remote-glue-folderchange.h"
#include "e-mail-remote-glue-msgiter.h"
#include "e-mail-remote-glue-utils.h"
#include "e-mail-remote-glue.h"

enum {
  SESSION_CHANGED_ADDED,
  SESSION_CHANGED_REMOVED,
  SESSION_CHANGED_CHANGED,
  SESSION_SHUTDOWN,
  STORE_CHANGED_ADDED,
  STORE_CHANGED_REMOVED,
  STORE_CHANGED_CHANGED,
  FOLDER_CHANGED_ADDED,
  FOLDER_CHANGED_REMOVED,
  FOLDER_CHANGED_CHANGED,
  LAST_SIGNAL
};

struct _EMailRemoteGluePrivate {

  Evolution_Mail_Session sess;
  EvolutionMailSessionListener* listener_sess;
  EvolutionMailStoreListener* listener_store;
  EvolutionMailFolderListener* listener_folder;
};

static void e_mail_remote_glue_class_init (EMailRemoteGlueClass *klass);
static void e_mail_remote_glue_init (EMailRemoteGlue *comp, EMailRemoteGlueClass *klass);
static void e_mail_remote_glue_finalize (GObject *object);

static guint e_mail_remote_signals[LAST_SIGNAL] = { 0 };

static GObjectClass *parent_class;


static void
emrg_session_changed_cb (EvolutionMailSessionListener* listener, 
			 Evolution_Mail_Session session,
			 Evolution_Mail_SessionChanges* changes, 
			 gpointer data)
{
  EMailRemoteGlue* emrg = data;
  EMailRemoteGlueSessionChange* emrg_sess_change = NULL;
  GSList* glue_sinfos = NULL;
  int sigtype = -1;
  int i;

  for (i = 0; i < changes->_length; i++) {
    emrg_sess_change = e_mail_remote_glue_sessionchange_new_from_EMSessionChange (changes->_buffer[i]);
    glue_sinfos = e_mail_remote_glue_sessionchange_get_store_info_list (emrg_sess_change);
    
    switch (changes->_buffer[i].type) {
    case Evolution_Mail_ADDED: 
      sigtype = SESSION_CHANGED_ADDED;
      break;

    case Evolution_Mail_REMOVED:
      sigtype = SESSION_CHANGED_REMOVED;
      break;

    case Evolution_Mail_CHANGED:
      sigtype = SESSION_CHANGED_CHANGED;
      break;
    }
    /* Object will be unref'ed immediately after emitting the signal.
       To keep the object around, one has to ref it in the signal-handler.
    */
    if (sigtype != -1) {
      g_signal_emit (emrg, e_mail_remote_signals[sigtype], 0, glue_sinfos);
      g_object_unref (emrg_sess_change);
    }
  }
}

static void
emrg_session_shutdown_cb (EvolutionMailSessionListener* listener, 
			  Evolution_Mail_Session session,
			  gpointer data)
{
  EMailRemoteGlue* emrg = data;
  g_signal_emit (emrg, e_mail_remote_signals[SESSION_SHUTDOWN], 0, session);
}

static void
emrg_store_changed_cb (EvolutionMailSessionListener* listener, 
		       Evolution_Mail_Store store,
		       Evolution_Mail_StoreChanges* changes, 
		       gpointer data)
{
  EMailRemoteGlue* emrg = data;
  EMailRemoteGlueStoreChange* emrg_storechange = NULL;
  GSList* glue_finfos = NULL;
  int sigtype = -1;
  int i;

  for (i = 0; i < changes->_length; i++) {
    emrg_storechange = e_mail_remote_glue_storechange_new_from_EMStoreChange (changes->_buffer[i]);
    glue_finfos = e_mail_remote_glue_storechange_get_folder_info_list (emrg_storechange);
    
    switch (changes->_buffer[i].type) {
    case Evolution_Mail_ADDED: 
      sigtype = STORE_CHANGED_ADDED;
      break;

    case Evolution_Mail_REMOVED:
      sigtype = STORE_CHANGED_REMOVED;
      break;

    case Evolution_Mail_CHANGED:
      sigtype = STORE_CHANGED_CHANGED;
      break;
    }
    /* Object will be unref'ed immediately after emitting the signal.
       To keep the object around, one has to ref it in the signal-handler.
    */
    if (sigtype != -1) {
      g_signal_emit (emrg, e_mail_remote_signals[sigtype], 0, glue_finfos);
      g_object_unref (emrg_storechange);
    }
  }
}

static void
emrg_folder_changed_cb (EvolutionMailSessionListener* listener, 
			Evolution_Mail_Folder folder,
			Evolution_Mail_FolderChanges* changes, 
			gpointer data)
{
  EMailRemoteGlue* emrg = data;
  EMailRemoteGlueFolderChange* emrg_folderchange = NULL;
  GSList* glue_msginfos = NULL;
  int sigtype = -1;
  int i;

  for (i = 0; i < changes->_length; i++) {
    emrg_folderchange = e_mail_remote_glue_folderchange_new_from_EMFolderChange (changes->_buffer[i]);
    glue_msginfos = e_mail_remote_glue_folderchange_get_message_info_list (emrg_folderchange);
    
    switch (changes->_buffer[i].type) {
    case Evolution_Mail_ADDED: 
      sigtype = FOLDER_CHANGED_ADDED;
      break;

    case Evolution_Mail_REMOVED:
      sigtype = FOLDER_CHANGED_REMOVED;
      break;

    case Evolution_Mail_CHANGED:
      sigtype = FOLDER_CHANGED_CHANGED;
      break;
    }
    /* Object will be unref'ed immediately after emitting the signal.
       To keep the object around, one has to ref it in the signal-handler.
    */
    if (sigtype != -1) {
      g_signal_emit (emrg, e_mail_remote_signals[sigtype], 0, glue_msginfos);
      g_object_unref (emrg_folderchange);
    }
  }
}

static void
emrg_initialize_callbacks (EMailRemoteGlue* emrg)
{
  EMailRemoteGluePrivate* priv = emrg->priv;
  
  g_signal_connect (G_OBJECT (priv->listener_sess), "changed", G_CALLBACK (emrg_session_changed_cb), emrg);
  g_signal_connect (G_OBJECT (priv->listener_sess), "shutdown", G_CALLBACK (emrg_session_shutdown_cb), emrg);
  g_signal_connect (G_OBJECT (priv->listener_store), "changed", G_CALLBACK (emrg_store_changed_cb), emrg);
  g_signal_connect (G_OBJECT (priv->listener_folder), "changed", G_CALLBACK (emrg_folder_changed_cb), emrg);
}

static void
emrg_initialize_session_and_listeners (EMailRemoteGlue* emrg)
{
	CORBA_Environment ev = { 0 };
	GNOME_Evolution_Component mail;
	GNOME_Evolution_Shell shell;
	EMailRemoteGluePrivate* priv = emrg->priv;

	shell = bonobo_activation_activate_from_id("OAFIID:GNOME_Evolution_Shell:2.4", 0, NULL, &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		e_mail_exception_dump(&ev, "activating shell");
		return;
	}

	do {
		mail = GNOME_Evolution_Shell_findComponent (shell, "mail", &ev);
		if (ev._major == CORBA_USER_EXCEPTION
		    && !strcmp(ev._id, ex_GNOME_Evolution_Shell_NotReady)) {
			CORBA_exception_free (&ev);
			printf("Shell not ready yet, waiting\n");
			sleep (1);
		} else if (ev._major != CORBA_NO_EXCEPTION) {
			e_mail_exception_dump (&ev, "finding mail component");
			CORBA_Object_release (shell, NULL);
			return;
		} else
			break;
	} while (1);

	priv->sess = Bonobo_Unknown_queryInterface (mail, "IDL:Evolution/Mail/Session:1.0", &ev);
	if (priv->sess == NULL || ev._major != CORBA_NO_EXCEPTION) {
		if (ev._major != CORBA_NO_EXCEPTION)
			e_mail_exception_dump(&ev, "querying for session interface");
		else
			printf("can't find session interface?\n");
		CORBA_Object_release(shell, NULL);
		CORBA_Object_release(mail, NULL);
		return;
	}

	printf("got session interface: %p\n", priv->sess);

	priv->listener_sess = evolution_mail_sessionlistener_new ();
	priv->listener_store = evolution_mail_storelistener_new ();
	priv->listener_folder = evolution_mail_folderlistener_new ();

	Evolution_Mail_Session_addListener(priv->sess, 
					   bonobo_object_corba_objref((BonoboObject *)priv->listener_sess), 
					   &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		e_mail_exception_dump(&ev, "adding session listener");
		return;
	}
	return;
}

/**
 * e_mail_remote_glue_get_type:
 *
 * Registers the #EMailRemoteGlue class if necessary, and returns the type ID
 * associated to it.
 *
 * Return value: The type ID of the #EMailRemoteGlue class.
 **/
GType
e_mail_remote_glue_get_type (void)
{
  static GType e_mail_remote_glue_type = 0;

  if (!e_mail_remote_glue_type) {
    static GTypeInfo info = {
      sizeof (EMailRemoteGlueClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) e_mail_remote_glue_class_init,
      NULL, NULL,
      sizeof (EMailRemoteGlue),
      0,
      (GInstanceInitFunc) e_mail_remote_glue_init
    };
    e_mail_remote_glue_type = g_type_register_static (G_TYPE_OBJECT, "EMailRemoteGlue", &info, 0);
  }

  return e_mail_remote_glue_type;
}

/* Class initialization function for the calendar-glue object */
static void
e_mail_remote_glue_class_init (EMailRemoteGlueClass *klass)
{
  GObjectClass *object_class;

  object_class = (GObjectClass *) klass;

  parent_class = g_type_class_peek_parent (klass);

  object_class->finalize = e_mail_remote_glue_finalize;

  e_mail_remote_signals[SESSION_CHANGED_ADDED] =
    g_signal_new ("session_changed_added",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (EMailRemoteGlueClass, session_changed_added),
		  NULL, NULL,
		  e_mail_remote_glue_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, G_TYPE_POINTER);

  e_mail_remote_signals[SESSION_CHANGED_REMOVED] =
    g_signal_new ("session_changed_removed",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (EMailRemoteGlueClass, session_changed_removed),
		  NULL, NULL,
		  e_mail_remote_glue_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, G_TYPE_POINTER);

  e_mail_remote_signals[SESSION_CHANGED_CHANGED] =
    g_signal_new ("session_changed_changed",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (EMailRemoteGlueClass, session_changed_changed),
		  NULL, NULL,
		  e_mail_remote_glue_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, G_TYPE_POINTER);

  e_mail_remote_signals[SESSION_SHUTDOWN] =
    g_signal_new ("session_shutdown",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (EMailRemoteGlueClass, session_shutdown),
		  NULL, NULL,
		  e_mail_remote_glue_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, G_TYPE_POINTER);

  e_mail_remote_signals[STORE_CHANGED_ADDED] =
    g_signal_new ("store_changed_added",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (EMailRemoteGlueClass, store_changed_added),
		  NULL, NULL,
		  e_mail_remote_glue_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, G_TYPE_POINTER);

  e_mail_remote_signals[STORE_CHANGED_REMOVED] =
    g_signal_new ("store_changed_removed",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (EMailRemoteGlueClass, store_changed_removed),
		  NULL, NULL,
		  e_mail_remote_glue_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, G_TYPE_POINTER);

  e_mail_remote_signals[STORE_CHANGED_CHANGED] =
    g_signal_new ("store_changed_changed",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (EMailRemoteGlueClass, store_changed_changed),
		  NULL, NULL,
		  e_mail_remote_glue_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, G_TYPE_POINTER);

  e_mail_remote_signals[FOLDER_CHANGED_ADDED] =
    g_signal_new ("folder_changed_added",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (EMailRemoteGlueClass, folder_changed_added),
		  NULL, NULL,
		  e_mail_remote_glue_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, G_TYPE_POINTER);

  e_mail_remote_signals[FOLDER_CHANGED_REMOVED] =
    g_signal_new ("folder_changed_removed",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (EMailRemoteGlueClass, folder_changed_removed),
		  NULL, NULL,
		  e_mail_remote_glue_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, G_TYPE_POINTER);

  e_mail_remote_signals[FOLDER_CHANGED_CHANGED] =
    g_signal_new ("folder_changed_changed",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (EMailRemoteGlueClass, folder_changed_changed),
		  NULL, NULL,
		  e_mail_remote_glue_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, G_TYPE_POINTER);
}

/* Object initialization function for the calendar-glue object */
static void
e_mail_remote_glue_init (EMailRemoteGlue *emrg, EMailRemoteGlueClass *klass)
{  
  emrg->priv = g_new0 (EMailRemoteGluePrivate, 1);
}

/**
 * e_mail_remote_glue_new:
 *
 * Creates a new empty calendar-glue component object.  
 *
 * Return value: A newly-created calendar-glue component object.
 **/
EMailRemoteGlue *
e_mail_remote_glue_new (void)
{
  return E_MAIL_REMOTE_GLUE (g_object_new (E_TYPE_MAIL_REMOTE_GLUE, NULL));
}

/* Finalize handler for the calendar-glue component object */
static void
e_mail_remote_glue_finalize (GObject *object)
{
  EMailRemoteGlue *comp;

  g_return_if_fail (object != NULL);
  g_return_if_fail (E_IS_MAIL_REMOTE_GLUE (object));

  comp = E_MAIL_REMOTE_GLUE (object);
  
  if (G_OBJECT_CLASS (parent_class)->finalize)
    (* G_OBJECT_CLASS (parent_class)->finalize) (object);
}

gboolean
e_mail_remote_glue_lookup_plugin_and_initialize (EMailRemoteGlue* emrg)
{
  EMailRemoteGluePrivate* priv = emrg->priv;
  Evolution_Mail_StoreInfos *stores;
  Evolution_Mail_FolderInfos *folders;
  CORBA_Environment ev = { 0 };
  int i;
  gboolean retval = TRUE;
  
  emrg_initialize_session_and_listeners (emrg);
  emrg_initialize_callbacks (emrg);

  stores = Evolution_Mail_Session_getStores (priv->sess, "", 
					     (Evolution_Mail_StoreListener)bonobo_object_corba_objref ((BonoboObject *)priv->listener_store), 
					     &ev);
  if (ev._major != CORBA_NO_EXCEPTION) {
    e_mail_exception_dump (&ev, "getting stores");
    retval = FALSE;
  }

  for (i = 0; i < stores->_length; i++) {
    
    Evolution_Mail_Store store = stores->_buffer[i].store;
    
    folders = Evolution_Mail_Store_getFolders (store, "",
					       (Evolution_Mail_FolderListener)bonobo_object_corba_objref ((BonoboObject *)priv->listener_folder),
					       &ev);
    if (ev._major != CORBA_NO_EXCEPTION) {
      e_mail_exception_dump (&ev, "getting stores");
      retval = FALSE;
      break;
    }
    
    CORBA_free (folders);
  }

  CORBA_free (stores);
  
  return retval;
}

GSList*
e_mail_remote_glue_get_stores (EMailRemoteGlue* emrg)
{
  EMailRemoteGluePrivate* priv = emrg->priv;
  Evolution_Mail_StoreInfos *stores;
  CORBA_Environment ev = { 0 };
  GSList* emrg_sinfo_list = NULL;
  int i;

  stores = Evolution_Mail_Session_getStores (priv->sess, "", 
					     (Evolution_Mail_StoreListener)bonobo_object_corba_objref ((BonoboObject *)priv->listener_store), 
					     &ev);
  if (ev._major != CORBA_NO_EXCEPTION) {
    e_mail_exception_dump (&ev, "getting stores");
    return NULL;
  }
  
  for (i = 0; i < stores->_length; i++) {
    EMailRemoteGlueStoreInfo* emrg_sinfo = e_mail_remote_glue_storeinfo_new_from_EMStoreInfo (stores->_buffer[i]);
    emrg_sinfo_list = g_slist_prepend (emrg_sinfo_list, emrg_sinfo);
  }
  if (emrg_sinfo_list)
    emrg_sinfo_list = g_slist_reverse (emrg_sinfo_list);
  return emrg_sinfo_list;
}

EvolutionMailSessionListener*
e_mail_remote_glue_peek_session_listener (EMailRemoteGlue* emrg)
{
  g_return_val_if_fail (emrg != NULL, NULL);
  g_return_val_if_fail (emrg->priv != NULL, NULL);
  
  return emrg->priv->listener_sess;
}

EvolutionMailStoreListener*
e_mail_remote_glue_peek_store_listener (EMailRemoteGlue* emrg)
{
  g_return_val_if_fail (emrg != NULL, NULL);
  g_return_val_if_fail (emrg->priv != NULL, NULL);
  
  return emrg->priv->listener_store;
}

EvolutionMailFolderListener*
e_mail_remote_glue_peek_folder_listener (EMailRemoteGlue* emrg)
{
  g_return_val_if_fail (emrg != NULL, NULL);
  g_return_val_if_fail (emrg->priv != NULL, NULL);
  
  return emrg->priv->listener_folder;
}
