/* Dazuko Interface. Interace with Dazuko for file access control.
   Written by John Ogness <jogness@antivir.de>

   Copyright (C) 2002-2003, H+BEDV Datentechnik GmbH
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   1. Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

   2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

   3. Neither the name of Dazuko nor the names of its contributors may be used
   to endorse or promote products derived from this software without specific
   prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/ioctl.h>
#include "dazukoio.h"

struct dazuko_t _GLOBAL_DAZUKO = {-1, -1};

int dazukoInitialize_TS(struct dazuko_t *dazuko)
{
	if (dazuko == NULL)
		return -1;

	dazuko->device = -1;
	dazuko->dev_major = -1;

	return 0;
}

int dazukoRegister(const char *groupName)
{
	return dazukoRegister_TS(&_GLOBAL_DAZUKO, groupName);
}

int dazukoRegister_TS(struct dazuko_t *dazuko, const char *groupName)
{
	struct option_t	*opt;
	char		buffer[10];
	int		err = 0;

	if (dazuko == NULL)
		return -1;

	if (groupName == NULL)
		groupName = "_GENERIC";

	if (dazuko->device < 0)
	{

		dazuko->device = open("/dev/dazuko", 0);
		if (dazuko->device < 0)
			return -1;

		memset(buffer, 0, sizeof(buffer));
		if (read(dazuko->device, buffer, sizeof(buffer)-1) < 1)
		{
			close(dazuko->device);
			dazuko->device = -1;
			return -1;
		}
		dazuko->dev_major = atoi(buffer);
	}

	opt = (struct option_t *)malloc(sizeof(struct option_t));
	if (opt == NULL)
	{
		close(dazuko->device);
		dazuko->device = -1;
		dazuko->dev_major = -1;
		return -1;
	}

	memset(opt, 0, sizeof(struct option_t));

	opt->command = REGISTER;
	strncpy(opt->buffer, groupName, sizeof(opt->buffer) - 1);
	opt->buffer_length = strlen(opt->buffer) + 1;

#ifdef __FreeBSD__
	if (ioctl(dazuko->device, IOCTL_SET_OPTION, &opt) != 0)
#else
	if (ioctl(dazuko->device, _IOW(dazuko->dev_major, IOCTL_SET_OPTION, void *), opt) != 0)
#endif
	{
		close(dazuko->device);
		dazuko->device = -1;
		dazuko->dev_major = -1;
		err = -1;
	}

	free(opt);

	return err;
}

int dazukoSetAccessMask(unsigned long accessMask)
{
	return dazukoSetAccessMask_TS(&_GLOBAL_DAZUKO, accessMask);
}

int dazukoSetAccessMask_TS(struct dazuko_t *dazuko, unsigned long accessMask)
{
	struct option_t	*opt;
	int		err = 0;

	if (dazuko == NULL)
		return -1;

	if (dazuko->device < 0 || dazuko->dev_major < 0)
		return -1;

	opt = (struct option_t *)malloc(sizeof(struct option_t));
	if (opt == NULL)
		return -1;

	memset(opt, 0, sizeof(struct option_t));

	opt->command = SET_ACCESS_MASK;
	opt->buffer[0] = (char)accessMask;
	opt->buffer_length = 1;

#ifdef __FreeBSD__
	if (ioctl(dazuko->device, IOCTL_SET_OPTION, &opt) != 0)
#else
	if (ioctl(dazuko->device, _IOW(dazuko->dev_major, IOCTL_SET_OPTION, void *), opt) != 0)
#endif
		err = -1;

	free(opt);

	return err;
}

static int dazuko_set_path(struct dazuko_t *dazuko, const char *path, int command)
{
	struct option_t	*opt;
	int		err = 0;

	if (dazuko == NULL)
		return -1;

	if (path == NULL)
		return -1;

	if (dazuko->device < 0 || dazuko->dev_major < 0)
		return -1;

	opt = (struct option_t *)malloc(sizeof(struct option_t));
	if (opt == NULL)
		return -1;

	memset(opt, 0, sizeof(struct option_t));

	opt->command = command;
	strncpy(opt->buffer, path, sizeof(opt->buffer) - 1);
	opt->buffer_length = strlen(opt->buffer) + 1;

#ifdef __FreeBSD__
	if (ioctl(dazuko->device, IOCTL_SET_OPTION, &opt) != 0)
#else
	if (ioctl(dazuko->device, _IOW(dazuko->dev_major, IOCTL_SET_OPTION, void *), opt) != 0)
#endif
		err = -1;

	free(opt);

	return err;
}

int dazukoAddIncludePath(const char *path)
{
	return dazukoAddIncludePath_TS(&_GLOBAL_DAZUKO, path);
}

int dazukoAddIncludePath_TS(struct dazuko_t *dazuko, const char *path)
{
	return dazuko_set_path(dazuko, path, ADD_INCLUDE_PATH);
}

int dazukoAddExcludePath(const char *path)
{
	return dazukoAddExcludePath_TS(&_GLOBAL_DAZUKO, path);
}

int dazukoAddExcludePath_TS(struct dazuko_t *dazuko, const char *path)
{
	return dazuko_set_path(dazuko, path, ADD_EXCLUDE_PATH);
}

int dazukoRemoveAllPaths(void)
{
	return dazukoRemoveAllPaths_TS(&_GLOBAL_DAZUKO);
}

int dazukoRemoveAllPaths_TS(struct dazuko_t *dazuko)
{
	struct option_t	*opt;
	int		err = 0;

	if (dazuko == NULL)
		return -1;

	if (dazuko->device < 0 || dazuko->dev_major < 0)
		return -1;

	opt = (struct option_t *)malloc(sizeof(struct option_t));
	if (opt == NULL)
		return -1;

	memset(opt, 0, sizeof(struct option_t));

	opt->command = REMOVE_ALL_PATHS;
	opt->buffer_length = 0;

#ifdef __FreeBSD__
	if (ioctl(dazuko->device, IOCTL_SET_OPTION, &opt) != 0)
#else
	if (ioctl(dazuko->device, _IOW(dazuko->dev_major, IOCTL_SET_OPTION, void *), opt) != 0)
#endif
		err = -1;

	free(opt);

	return err;

}

int dazukoGetAccess(struct access_t *acc)
{
	return dazukoGetAccess_TS(&_GLOBAL_DAZUKO, acc);
}

int dazukoGetAccess_TS(struct dazuko_t *dazuko, struct access_t *acc)
{
	if (dazuko == NULL)
		return -1;

	if (acc == NULL)
		return -1;

	if (dazuko->device < 0 || dazuko->dev_major < 0)
		return -1;

	memset(acc, 0, sizeof(struct access_t));

#ifdef __FreeBSD__
	if (ioctl(dazuko->device, IOCTL_GET_AN_ACCESS, &acc) != 0)
#else
	if (ioctl(dazuko->device, _IOR(dazuko->dev_major, IOCTL_GET_AN_ACCESS, struct access_t *), acc) != 0)
#endif
		return -1;

	return 0;
}

int dazukoReturnAccess(struct access_t *acc)
{
	return dazukoReturnAccess_TS(&_GLOBAL_DAZUKO, acc);
}

int dazukoReturnAccess_TS(struct dazuko_t *dazuko, struct access_t *acc)
{
	if (dazuko == NULL)
		return -1;

	if (acc == NULL)
		return -1;

	if (dazuko->device < 0 || dazuko->dev_major < 0)
		return -1;

#ifdef __FreeBSD__
	if (ioctl(dazuko->device, IOCTL_RETURN_ACCESS, &acc) != 0)
#else
	if (ioctl(dazuko->device, _IOW(dazuko->dev_major, IOCTL_RETURN_ACCESS, struct access_t *), acc) != 0)
#endif
		return -1;

	return 0;
}

int dazukoUnregister(void)
{
	return dazukoUnregister_TS(&_GLOBAL_DAZUKO);
}

int dazukoUnregister_TS(struct dazuko_t *dazuko)
{
	int		error = -1;
	struct option_t	*opt;

	if (dazuko == NULL)
		return -1;

	if (dazuko->device >= 0)
	{
		opt = (struct option_t *)malloc(sizeof(struct option_t));
		if (opt == NULL)
		{
			close(dazuko->device);
			dazuko->device = -1;
			dazuko->dev_major = -1;
			return -1;
		}

		memset(opt, 0, sizeof(struct option_t));

		opt->command = UNREGISTER;
		opt->buffer_length = 0;

#ifdef __FreeBSD__
		if (ioctl(dazuko->device, IOCTL_SET_OPTION, &opt) != 0)
#else
		if (ioctl(dazuko->device, _IOW(dazuko->dev_major, IOCTL_SET_OPTION, void *), opt) != 0)
#endif
		{
			close(dazuko->device);
			dazuko->device = -1;
			dazuko->dev_major = -1;
			error = -1;
		}
		else
		{
			error = 0;
		}

		free(opt);

		if (error == 0)
			error = close(dazuko->device);
	}

	dazuko->device = -1;
	dazuko->dev_major = -1;

	return error;
}
