/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.test.remoting.transport.multiplex;

import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.util.HashMap;

import junit.framework.TestCase;

import org.jboss.logging.Logger;
import org.jboss.remoting.Client;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.callback.Callback;
import org.jboss.remoting.callback.HandleCallbackException;
import org.jboss.remoting.callback.InvokerCallbackHandler;
import org.jboss.remoting.transport.Connector;
import org.jboss.remoting.transport.multiplex.Multiplex;

/**
 * This test verifies that a <code>MultiplexClientInvoker</code> that is connected
 * to a non-responsive server can reconnect to a new server at the same address.
 * <p>
 * @author <a href="mailto:ron.sigal@jboss.com">Ron Sigal</a>
 * <p>
 * Copyright (c) Jul 21, 2006
 * </p>
 */
public class RestartServerTestClient extends TestCase
{
   private static Logger log = Logger.getLogger(RestartServerTestClient.class);
   
   
   public void testServerRestart()
   {
      log.info("entering " + getName());
      
      try
      {
         Socket s = new Socket("localhost", RestartServerTestServer.ssPort);
         InputStream is = s.getInputStream();
         OutputStream os = s.getOutputStream();
         is.read();
         
         HashMap cconfig = new HashMap();
         cconfig.put("timeout", "1000");
         cconfig.put(Multiplex.CLIENT_MULTIPLEX_ID, "id");
         Client client = new Client(new InvokerLocator(RestartServerTestServer.locatorURI), cconfig);
         
         HashMap sconfig = new HashMap();
         sconfig.put(Multiplex.SERVER_MULTIPLEX_ID, "id");
         Connector connector = new Connector("multiplex://localhost", sconfig);
         connector.create();
         connector.start();
         client.connect();
         
         // Test method invocation on first server.
         assertEquals(new Integer(3), client.invoke("abc"));
         
         // Test callbacks on first server.
         InvokerLocator callbackLocator = connector.getLocator();
         SampleCallbackHandler callbackHandler = new SampleCallbackHandler();
         client.addListener(callbackHandler, callbackLocator);
         Thread.sleep(1000);
         assertEquals(new Integer(3), callbackHandler.callback);

         os.write(1);
         is.read();
         
         // Test method invocation on second server.
         try
         {
            assertEquals(new Integer(7), client.invoke("xyz"));
         }
         catch (Throwable t)
         {
            log.info("Client.invoke(\"xyz\") failed: trying one more time");
            assertEquals(new Integer(7), client.invoke("xyz"));
         }
         
         // Test callbacks on first server.
         callbackLocator = connector.getLocator();
         callbackHandler = new SampleCallbackHandler();
         client.addListener(callbackHandler, callbackLocator);
         Thread.sleep(1000);
         assertEquals(new Integer(7), callbackHandler.callback);
         
         os.write(3);
         client.disconnect();
         connector.stop();
         s.close();
         
         log.info(getName() + " PASSES");
      }
      catch (Throwable t)
      {
         t.printStackTrace();
         log.error(getName() + " FAILS");
         fail();
      }
   }
   
   
   
   public static class SampleCallbackHandler implements InvokerCallbackHandler
   {
      Object callback;
      
      public void handleCallback(Callback callback) throws HandleCallbackException
      {
         System.out.println("Received callback value of: " + callback.getCallbackObject());
         this.callback = callback.getParameter();
      }
   }
}
