/* CliqueNumber.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.standard;

import java.util.Arrays;

import org.grinvin.graphs.GraphBundleView;
import org.grinvin.invariants.computers.AbstractInvariantComputer;
import org.grinvin.invariants.values.IntegerValue;

/**
 * Invariant computer which returns the clique number.
 */
public class CliqueNumber extends AbstractInvariantComputer {

    public IntegerValue compute(GraphBundleView bundle) {
        
        int[][] nonAdjacencyList = nonAdjacencyList (bundle);
        int n = nonAdjacencyList.length;
        boolean [] allowed = new boolean [n];
        Arrays.fill (allowed, true);
        int maxsize = findCliqueNumber(0, allowed, n, nonAdjacencyList, 0);
        return new IntegerValue(maxsize, this);
    }

    
    //
    private int[][] nonAdjacencyList (GraphBundleView bundle) {
        boolean [][] ad = bundle.booleanAdjacencyMatrix();
        int n = ad.length;
        // for every vertex, the list of vertices NOT adjacent to it
        int[][] nonAdjacencyList = new int[n][];
        for (int v=0; v < n; v++) {
            int cnt = -1;
            for (boolean b : ad[v])
                if (!b)
                    cnt ++;
            int[] nonad = new int[cnt];
            cnt = 0;
            int j;
            for (j=0; j < v; j++)
                if (!ad[v][j]) {
                    nonad[cnt] = j;
                    cnt ++;
                }
            for (j++; j < n; j++)
                if (!ad[v][j]) {
                    nonad[cnt] = j;
                    cnt ++;
                }
            nonAdjacencyList[v] = nonad;
        }       
        return nonAdjacencyList;
    }
    
    /** */
    private int findCliqueNumber(int size, boolean[] allowed, int numberAllowed, 
            int[][] nonAdjacencyList, int maxsize) {
        if (numberAllowed == 0) {
            if (size > maxsize)
                return size;
        } 
        else if (numberAllowed + size < maxsize)
            return maxsize; // it will never be  possible to find a better maxsize 
        else {
            int v = 0;
            while (numberAllowed != 0) {
                while (!allowed[v])
                    v++;
                allowed[v] = false;
                numberAllowed --;
                
                // remove neighbours of v from allowed
                boolean[] newAllowed = allowed.clone();
                int newNumberAllowed = numberAllowed;
                for (int nb : nonAdjacencyList[v])
                    if (newAllowed[nb]) {
                        newAllowed[nb] = false;
                        newNumberAllowed --;
                    }
                maxsize = findCliqueNumber(size+1,newAllowed, newNumberAllowed, 
                            nonAdjacencyList, maxsize);
            }
        }
        return maxsize;
    }
    
    public String getInvariantId() {
        return "org.grinvin.invariants.CliqueNumber";
    }
    
    
}
