; ****************************************************************************
;
;  LDLINUX.ASM
;
;  A program to boot Linux kernels off an MS-DOS formatted floppy disk.  This
;  functionality is good to have for installation floppies, where it may
;  be hard to find a functional Linux system to run LILO off.
;
;  This program allows manipulation of the disk to take place entirely
;  from MS-LOSS, and can be especially useful in conjunction with the
;  umsdos filesystem.
;
;  This file is loaded in stages; first the boot sector at offset 7C00h,
;  then the first sector (cluster, really, but we can only assume 1 sector)
;  of LDLINUX.SYS at 7E00h and finally the remainder of LDLINUX.SYS at 8000h
;
;   Copyright (C) 1994-95  H. Peter Anvin
;
;   This code is free software under the terms of the GNU General Public
;   License, version 2, or at your option any later version.
;
; ****************************************************************************

                page 79
;                page 59

;                jumps                   ; Assemble with /m option!!!!
                smart                   ; Optimizes some for us
;
; Some semi-configurable constants... change on your own risk
;
max_cmd_len     equ 2047                ; Must be odd; 2047 is the kernel limit
setup_secs      equ 4                   ; Must be same as kernel
real_mode_secs  equ setup_secs+1
retry_count     equ 6                   ; How patient are we with the disk?

;
; Should be updated with every release to avoid bootsector/SYS file mismatch
;
version_str     equ '1.20'              ; Must be 4 characters long!!!
;debug           equ 1                   ; Uncomment to enable debugging

;
; Segments used by Linux
;
real_mode_seg   segment at 9000h        ; Real-mode Linux code
		org 20h
kern_cmd_magic  dw ?                    ; Magic # for command line
kern_cmd_offset dw ?                    ; Offset for kernel command line
		org 508
kern_def_root   dw ?                    ; Default root device
kern_boot_magic dw ?                    ; Must be 0AA55h
		org 4000h-12            ; Were bootsect.S puts it...
linux_stack     label byte
linux_fdctab    label byte
		org 4000h
cmd_line_here   equ $                   ; Should be out of the way
real_mode_seg   ends

setup_seg       segment at 9020h        ; The setup code wants to be run
		org 0h                  ; as 9020:0000, not 9000:0200
setup_entry     equ $
setup_seg       ends

load_here_seg   segment at 1000h-real_mode_secs*20h ; Kernel is loaded here
		org 0h
real_mode_load  equ $
		org real_mode_secs*512
prot_mode_code  equ $
load_here_seg   ends
;
; Our segment (is always at 0000h, unless we're debugging.)
;
_text           segment para 'code'
		assume cs:_text
		org 0h
zero_offset     equ $

		org 4*1Eh               ; In the interrupt table
fdctab          label dword
fdctab1         dw ?
fdctab2         dw ?

		org 0100h
bogus           label far               ; To keep TLINK or EXE2BIN happy
;
; Hook for debugger stuff.  This gets automatically removed when
; generating the real thing.
;
; Initialize the registers for debugger operation
;
		assume ds:_text, es:_text, ss:_text
		cli
		mov ax,cs
		mov ds,ax
		mov es,ax
		mov ss,ax
		mov sp,offset StackBuf
		sti
                cld
;
; Load the actual boot sector so we can copy the data block
;
		xor ax,ax               ; Reset floppy
		xor dx,dx
		int 13h
		mov cx,6                ; Retry count...
debug_tryloop:  push cx
		mov bx,offset trackbuf
		mov cx,0001h
		xor dx,dx
		mov ax,0201h
		int 13h
		pop cx
		jnc debug_okay
		loop debug_tryloop
		int 3                   ; Halt! (Breakpoint)
debug_okay:     mov si,(offset trackbuf)+0bh
		mov di,offset bsBytesPerSec
		mov cx,33h
		rep movsb
;
; Save bogus "BIOS floppy block" info to the stack in case we hit kaboom
;
		push si
		push si
		push si                 ; Writing to the trackbuf is harmless
;
; A NOP where we can breakpoint, then jump into the code *after*
; the segment register initialization section
;
		nop
		jmp debugentrypt

		org 0600h
trackbuf        label byte              ; Track buffer goes here
trackbufsize    equ 16384               ; Safe size of track buffer

		org 7B00h               ; Here we keep our BSS stuff
StackBuf        equ $                   ; Start the stack here (grow down)
RootDir         label dword             ; Location of root directory
RootDir1        dw ?
RootDir2        dw ?
DataArea        label dword             ; Location of data area
DataArea1       dw ?
DataArea2       dw ?
FBytes          label dword             ; Used by open/getc
FBytes1         dw ?
FBytes2         dw ?
RootDirSize     dw ?                    ; Root dir size in sectors
DirScanCtr      dw ?                    ; Used while searching directory
DirBlocksLeft   dw ?                    ; Ditto
EndofDirSec     dw ?                    ; = trackbuf+bsBytesPerSec-31
RunLinClust     dw ?                    ; Cluster # for LDLINUX.SYS
ClustSize       dw ?                    ; Bytes/cluster
SecPerClust     dw ?                    ; Same as bsSecPerClust, but a word
BufSafe         dw ?                    ; Clusters we can load into trackbuf
BufSafeSec      dw ?                    ; = how many sectors?
BufSafeBytes    dw ?                    ; = how many bytes?
EndOfGetCBuf    dw ?                    ; = getcbuf+BufSafeBytes
KernelClust     dw ?                    ; Clusters needed at 10000h
ClustPerMoby    dw ?                    ; Clusters per 64K
FClust          dw ?                    ; Number of clusters in open/getc file
FNextClust      dw ?                    ; Pointer to next cluster in d:o
FPtr            dw ?                    ; Pointer to next char in buffer
RetryCount      db ?                    ; Used for disk access retries
KbdFlags        db ?                    ; Check for keyboard escapes
MNameBuf        db 11 dup(?)            ; Mangled file name buffer
FKeyName        db 10*16 dup(?)         ; File names for F-key help
		org 7C00h
bootsec         equ $
		jmp short start
		nop
;
; "Superblock" follows -- it's in the boot sector, so it's already
; loaded and ready for us
;
bsOemName       db 'LDLINUX '           ; The SYS command sets this, so...
bsBytesPerSec   dw ?
bsSecPerClust   db ?
bsResSectors    dw ?
bsFATs          db ?
bsRootDirEnts   dw ?
bsSectors       dw ?
bsMedia         db ?
bsFATsecs       dw ?
bsSecPerTrack   dw ?
bsHeads         dw ?
bsHiddenSecs    label dword
bsHidden1       dw ?
bsHidden2       dw ?
bsHugeSectors   label dword
bsHugeSec1      dw ?
bsHugeSec2      dw ?
bsDriveNumber   db ?
bsReserved1     db ?
bsBootSignature db ?                    ; 29h if the following fields exist
bsVolumeID      dd ?
bsVolumeLabel   db 11 dup(?)
bsFileSysType   db 8 dup(?)             ; Must be FAT12 for this version
;
; Note we don't check the constraints above now; we did that at install
; time (we hope!)
;

start           label near
floppy_table    label byte              ; No sense in wasting memory
		cli                     ; No interrupts yet, please
		xor ax,ax
		mov es,ax
		mov ss,ax
		mov sp,offset StackBuf  ; Just below BSS
		assume es:_text, ss:_text, ds:NOTHING
;
; Now sautee the BIOS floppy info block to that it will support decent-
; size transfers; the floppy block is 11 bytes and is stored in the
; INT 1Eh vector (brilliant waste of resources, eh?)
;
; Of course, if BIOSes had been properly programmed, we wouldn't have
; had to waste precious boot sector space with this code.
;
		mov bx,offset fdctab
		lds si,ss:[bx]          ; DS:SI -> original
		push ds                 ; Save on stack in case
		push si                 ; we have to bail
		push bx
		mov cx,6                ; 12 bytes
		mov di,offset floppy_table
		push di
		cld
		rep movsw               ; Faster to move words
		pop di
		mov ds,ax               ; Now we can point DS to here, too
		assume ds:_text
		mov cl,byte ptr bsSecPerTrack ; Patch the sector count
		mov [di+4],cl
;                mov byte ptr [di+9],0Fh ; I have no clue what this does???
		mov [bx+2],ax           ; Segment 0
		mov [bx],di             ; offset floppy_block
;
; Ready to enable interrupts, captain
;
		sti
;
; Reset floppy system to load new parameter block
;
		xor dx,dx
                int 13h                 ; AH = 00h already
;
; Now we have to do some arithmetric to figure out where things are located.
; If Microsoft had had brains they would already have done this for us,
; and stored it in the superblock at format time, but here we go,
; wasting precious boot sector space again...
;
debugentrypt:
                mov al,bsFATs           ; Number of FATs
                jc kaboom               ; If the floppy init failed
					; (too far to be above the mov)
                cbw                     ; Clear AH
                mul bsFATsecs           ; Get the size of the FAT area
		add ax,bsHidden1        ; Add hidden sectors
		adc dx,bsHidden2
		add ax,bsResSectors     ; And reserved sectors (why two?)
		adc dx,0

		mov RootDir1,ax         ; Location of root directory
		mov RootDir2,dx
		mov DataArea1,ax
		mov DataArea2,dx
		push ax
		push dx

		mov ax,32               ; Size of a directory entry
		mul bsRootDirEnts
		mov bx,bsBytesPerSec
		add ax,bx               ; Round up, not down
		dec ax
		div bx                  ; Now we have the size of the root dir
		mov RootDirSize,ax
		mov DirScanCtr,ax
		add bx,(offset trackbuf)-31
		mov EndofDirSec,bx      ; End of a single directory sector

		add DataArea1,ax        ; Now we have the location of the
		adc DataArea2,0         ; first data cluster
;
; Now the fun begins.  We have to search the root directory for
; LDLINUX.SYS and load the first sector, so we have a little more
; space to have fun with.  Then we can go chasing through the FAT.
; Joy!!
;
sd_nextsec:     pop dx
		pop ax
		push ax
		push dx
		mov bx,offset trackbuf
		call getonesec
		mov si,offset trackbuf
sd_nextentry:   cmp byte ptr [si],0     ; Directory high water mark
jz_kaboom:      jz kaboom
		mov di,offset ldlinux_sys
		mov cx,11
		push si
		repe cmpsb
		pop si
		je found_it
		add si,32               ; Distance to next
		cmp si,EndofDirSec
		jb sd_nextentry
		add ax,1
		adc dx,0
		dec DirScanCtr
		jnz sd_nextsec
;
; kaboom: write a message and bail out.
;
kaboom          proc near
		mov si,offset bailmsg
		call writestr           ; Returns with AL = 0
		cbw                     ; Sets AH = 0 (shorter than XOR)
		int 16h                 ; Wait for keypress
		mov sp,(offset StackBuf)-2*3 ; Reset stack
		pop si                  ; BIOS floppy block address
		cli
		pop [si]                ; Restore location
		pop [si+2]
		sti
		int 19h                 ; And try once more to boot...
norge:          jmp short norge         ; If int 19h returned... oh boy...
kaboom          endp
;
; found_it: now we compute the location of the first sector, then
;           load it and JUMP (since we're almost out of space)
;
found_it:       pop ax
		pop ax
		mov al,bsSecPerClust
		cbw                     ; We won't have 128 sec/cluster
		mov bp,ax               ; Load an entire cluster
		mov bx,[si+26]          ; First cluster
		push bx                 ; Remember which cluster it was
		dec bx                  ; First cluster is cluster 2
		dec bx
		mul bx
		add ax,DataArea1
		adc dx,DataArea2
		mov bx,offset ldlinux_magic
		call getlinsec
		mov si,offset bs_magic
		mov di,offset ldlinux_magic
		mov cx,magic_len
		repe cmpsb              ; Make sure that the bootsector
		jne kaboom
		jmp ldlinux_ent         ; matches LDLINUX.SYS
;
; writestr: write a null-terminated string to the console
;
writestr        proc near
wstr_1:         lodsb
		and al,al
		jz return
		mov ah,0Eh              ; Write to screen as TTY
		mov bx,0007h            ; White on black, current page
		int 10h
		jmp short wstr_1
writestr        endp
;
; disk_error: decrement the retry count and bail if zero
;
disk_error:     dec si                  ; SI holds the disk retry counter
		jz kaboom
		xchg ax,bx              ; Shorter than MOV
		pop bx
		pop cx
		pop dx
		jmp disk_try_again
;
; getonesec: like getlinsec, but pre-sets the count to 1
;
getonesec       proc near
		mov bp,1
		; Fall through to getlinsec
getonesec       endp
;
; getlinsec: load a sequence of BP floppy sector given by the linear sector
;            number in DX:AX into the buffer at ES:BX.  We try to optimize
;            by loading up to a whole track at a time, but the user
;            is responsible for not crossing a 64K boundary.
;            (Yes, BP is weird for a count, but it was available...)
;
;            On return, BX points to the first byte after the transferred
;            block.
;
getlinsec       proc near
		mov si,bsSecPerTrack
		div si                  ; Convert linear to sector/track
		mov cx,dx               ; Save sector
		xor dx,dx               ; 32-bit track number
		div bsHeads             ; Convert track to head/cyl
		;
		; Now we have AX = cyl, DX = head, CX = sector (0-based)
		; for the very first sector, SI = bsSecPerTrack
		;
gls_nexttrack:  push si
		push bp
		sub si,cx               ; Sectors left on track
		cmp bp,si
		jna gls_lasttrack
		mov bp,si               ; No more than a trackful, please!
gls_lasttrack:  push ax                 ; Cylinder #
		push dx                 ; Head #
		push bp                 ; Number of sectors we're transferring

		push cx
		mov cl,6                ; Because IBM was STOOPID
		shl ah,cl               ; and thought 8 bits was enough
					; then thought 10 bits was enough...
		pop cx                  ; Sector #
		inc cx                  ; Sector numbers are 1-based
		or cl,ah
		mov ch,al
		mov dh,dl
		mov dl,bsDriveNumber
		xchg ax,bp              ; Sector to transfer count
					; (xchg shorter than mov)
		mov ah,02h              ; Read it!
;
; Do the disk transfer... save the registers in case we fail :(
;
		mov si,retry_count      ; # of times to retry a disk access
disk_try_again: push dx
		push cx
		push bx
		push ax
		push si
		int 13h
		pop si
		pop bx
		jc disk_error
;
; It seems the following test fails on some machines (buggy BIOS?)
;
;                cmp al,bl               ; Check that we got what we asked for
;                jne disk_error
;
; Disk access successful
;
		pop bx                  ; Buffer location
		pop si                  ; Not needed anymore
		pop si                  ; Neither is this
		pop si                  ; Sector transferred count
		mov ax,si               ; Reduce sector left count
		mul bsBytesPerSec       ; Figure out how much to advance ptr
		add bx,ax               ; Update buffer location
		pop dx                  ; Head #
		pop ax                  ; Cyl #
		inc dx                  ; Next track on cyl
		cmp dx,bsHeads          ; Was this the last one?
		jb gls_nonewcyl
		inc ax                  ; If so, new cylinder
                xor dx,dx               ; First head on new cylinder
gls_nonewcyl:   pop bp                  ; Sectors left to transfer
		xor cx,cx               ; First sector on new track
		sub bp,si               ; Reduce with # of sectors just read
		pop si
		ja gls_nexttrack
return:         ret
getlinsec       endp

bailmsg         db 'Boot failed: change disks and press any key', 0Dh, 0Ah, 0

bs_checkpt      equ $                   ; Must be <= 7DE5h

		org 7DE5h
bs_magic        label byte              ; The following 32 bytes should
					; match ldlinux_magic
ldlinux_sys     db 'LDLINUX SYS'        ; Looks like this in the root dir
		db ' '
bs_version      db version_str
		db ' '
bs_date         db ??date               ; 8 bytes date
magic_len       equ $-bs_magic

bootsignature   dw 0AA55h

;
; ===========================================================================
;  End of boot sector
; ===========================================================================
;  Start of LDLINUX.SYS
; ===========================================================================
;
; Put the FAT right after the code, aligned on a sector boundary
;
FAT             equ (ldlinux_end-zero_offset+511) and 0FE00h
;
; Put getc buffer right after FAT (the FAT buffer is 6K, the max size
; of a 12-bit FAT)
;
getcbuf         equ FAT+6*1024
;
; This "magic number" works well with the "type" command... :-)
;
ldlinux_magic   db 'LDLINUX'
missing_dot     db ' '
		db 'SYS ', version_str, ' ', ??date
magic_eof       db 0Dh, 0Ah, 01Ah

		org 7E20h
ldlinux_ent     label near
;
; The boot sector left the cluster number of this first LDLINUX.SYS
; sector on the stack.  We'll need it later, so we should pop it off
;
		pop RunLinClust
;
; Tell the user we got this far
;
		mov si,offset crlf
		call writestr
		mov missing_dot,'.'
		mov magic_eof,0
		mov si,offset ldlinux_magic
		call writestr
;
; Remember, the boot sector loaded only the first cluster of LDLINUX.SYS.
; We can really only rely on a single sector having been loaded.  Hence
; we should load the FAT into RAM and start chasing pointers...
;
		mov bx,offset FAT               ; Where it goes in memory
		mov ax,bsHidden1                ; Hidden sectors
		mov dx,bsHidden2
		add ax,bsResSectors             ; plus reserved sectors = FAT
		adc dx,0
		mov bp,bsFATsecs                ; Sectors/FAT
		call getlinsec                  ; Load it in...
;
; Fine, now we have the FAT in memory.  How big is a cluster, really?
; Also figure out how many clusters will fit in an 8K buffer, and how
; many sectors and bytes that is
;
		mov al,bsSecPerClust            ; We do this in the boot
		xor ah,ah                       ; sector, too, but there
		mov SecPerClust,ax              ; wasn't space to save it
		mul bsBytesPerSec
		mov ClustSize,ax                ; Bytes/cluster
		mov bx,ax
                mov ax,trackbufsize
		xor dx,dx
		div bx
                mov BufSafe,ax                  ; # of cluster in trackbuf
		mul SecPerClust
		mov BufSafeSec,ax
		mul bsBytesPerSec
		mov BufSafeBytes,ax
                add ax,getcbuf                  ; getcbuf is same size as
                mov EndOfGetCBuf,ax             ; trackbuf, for simplicity
;
; Now we read the rest of LDLINUX.SYS.  Don't bother loading the first
; cluster again, though.
;
		mov bx,offset ldlinux_magic
		add bx,ClustSize
		mov si,RunLinClust
		call nextcluster
		xor dx,dx
		mov ax,ldlinux_len-1            ; To be on the safe side
		add ax,ClustSize
		div ClustSize                   ; the number of clusters
		dec ax                          ; We've already read one
                jz all_read_jmp
		mov cx,ax
		call getfssec
;
; All loaded up
;
all_read_jmp:
                mov si,offset copyright_str
                call writestr
                jmp all_read
;
; -----------------------------------------------------------------------------
; Subroutines that have to be in the first sector
; -----------------------------------------------------------------------------
;
; getfssec: Get multiple clusters from a file, given the starting cluster.
;
;       This routine makes sure the subtransfers do not cross a 64K boundary,
;       and will correct the situation if it does, UNLESS *sectors* cross
;       64K boundaries.
;
;       ES:BX   -> Buffer
;       SI      -> Starting cluster number (2-based)
;       CX      -> Cluster count (0FFFFh = until end of file)
;
						; 386 check
getfssec        proc near
getfragment:    xor bp,bp                       ; Fragment sector count
		mov ax,si                       ; Get sector address
		dec ax                          ; Convert to 0-based
		dec ax
		mul SecPerClust
		add ax,DataArea1
		adc dx,DataArea2
getseccnt:                                      ; See if we can read > 1 clust
		add bp,SecPerClust
		dec cx                          ; Reduce clusters left to find
		mov di,si                       ; Predict next cluster
		inc di
		call nextcluster
		jc gfs_eof                      ; At EOF?
		jcxz endfragment                ; Or was it the last we wanted?
		cmp si,di                       ; Is file continuous?
		jz getseccnt                    ; Yes, we can get
endfragment:    clc                             ; Not at EOF
gfs_eof:        pushf                           ; Remember EOF or not
		push si
		push cx
gfs_getchunk:
		push ax
		push dx
		mov ax,es                       ; Check for 64K boundaries.
		shl ax,1                        ; This really belongs in
		shl ax,1                        ; getlinsec, but it would
		shl ax,1                        ; make it not fit in the boot
		shl ax,1                        ; sector.
		add ax,bx
                xor dx,dx
		neg ax
                jnz gfs_partseg
                inc dx                          ; Full 64K segment
gfs_partseg:
                div bsBytesPerSec               ; How many sectors fit?
		mov si,bp
		sub si,ax                       ; Compute remaining sectors
		jbe gfs_lastchunk
		mov bp,ax
		pop dx
		pop ax
		push si                         ; Save remaining sector count
		push ax                         ; Save position
		push dx
		push bp                         ; Save sectors to transfer
		call getlinsec
		pop bp
		pop dx
		pop ax
		add ax,bp                       ; Advance sector pointer
		adc dx,0
		pop bp                          ; Load remaining sector counter
		jmp gfs_getchunk
gfs_lastchunk:  pop dx
		pop ax          
		call getlinsec
		pop cx
		pop si
		popf
		jcxz gfs_return                 ; If we hit the count limit
		jnc getfragment                 ; If we didn't hit EOF
gfs_return:     ret
getfssec        endp
;
; nextcluster: Advance a cluster pointer in SI to the next cluster
;              pointed at in the FAT tables (note: FAT12 assumed)
;              Sets CF on return if end of file.
;
nextcluster     proc near
		push bx
		mov bx,si                       ; Multiply by 3/2
		shr bx,1
		pushf                           ; CF now set if odd
		add si,bx
		mov si,word ptr FAT[si]
		popf
		jnc nc_even
		shr si,1                        ; Needed for odd only
		shr si,1
		shr si,1
		shr si,1
nc_even:
		and si,0FFFh
		cmp si,0FF0h                    ; Clears CF if at end of file
		cmc                             ; But we want it SET...
		pop bx
nc_return:      ret
nextcluster     endp

;
; Debug routine
;
		ifdef debug
safedumpregs    proc near
                cmp Debug_Magic,0D00Dh
                jnz nc_return
                jmp dumpregs
safedumpregs    endp
		endif

;
; Data that has to be in the first sector
;
copyright_str   db '  Copyright (C) 1994-95 H. Peter Anvin'
crlf            db 0Dh, 0Ah, 0

rl_checkpt      equ $                           ; Must be <= 8000h

; ----------------------------------------------------------------------------
;  End of code and data that have to be in the first sector
; ----------------------------------------------------------------------------

all_read        label near
;
; Check that no moron is trying to boot Linux on a 286 or so.  According
; to Intel, the way to check is to see if the high 4 bits of the FLAGS
; register are either all stuck at 1 (8086/8088) or all stuck at 0
; (286 in real mode), if not it is a 386 or higher.  They didn't
; say how to check for a 186/188, so I *hope* it falls out as a 8086
; or 286 in this test.
;
; Also, provide an escape route in case it doesn't work.
;
check_escapes:
		mov ah,02h                      ; Check keyboard flags
		int 16h
		mov KbdFlags,al                 ; Save for boot prompt check
		test al,04h                     ; Ctrl->skip 386 check
		jnz skip_checks
test_8086:
		pushf                           ; Get flags
		pop ax
		and ax,0FFFh                    ; Clear top 4 bits
		push ax                         ; Load into FLAGS
		popf
		pushf                           ; And load back
		pop ax
		and ax,0F000h                   ; Get top 4 bits
		cmp ax,0F000h                   ; If set -> 8086/8088
		je not_386
test_286:
		pushf                           ; Get flags
		pop ax
		or ax,0F000h                    ; Set top 4 bits
		push ax
		popf
		pushf
		pop ax
		and ax,0F000h                   ; Get top 4 bits
		jnz is_386                      ; If not clear -> 386
not_386:
		mov si,offset err_not386
		call writestr
		jmp kaboom
is_386:
		.386                            ; Now we know it's a 386
;
; Now check that there is at least 608K of low (DOS) memory
; (608K = 9800h segments)
;
		int 12h
                cmp ax,608
		jae enough_ram
		mov si,offset err_noram
		call writestr
		jmp kaboom
enough_ram:
skip_checks:
;
; Now we're all set to start with our *real* business.  First load the
; configuration file (if any) and parse it.
;
; Note: Even though we are on a 386+, I haven't used 32-bit registers
; (even though I *really* wanted to) because according to the author
; of PKZIP, there are 386 BIOSes that garble the upper 16 bits when
; they serve an interrupt!!!!!!!  He found out the hard way.  BIG
; thumbs down for certain BIOS manufacturers!
;
                mov si,offset linuxauto_cmd     ; Default command: "linux auto"
                mov di,offset default_cmd
                mov cx,11
                rep movsb
;
; Load configuration file
;
                mov di,offset syslinux_cfg
                call open
                jz no_config_file
parse_config:
                call getkeyword
                jc no_config_file               ; Config file loaded
                cmp ax,'ed'                     ; DEfault (reversed due to
                                                ; screwy assembler)
                je pc_default
                cmp ax,'pa'                     ; APpend
                je pc_append
                cmp ax,'it'                     ; TImeout
                je pc_timeout
                cmp ax,'rp'                     ; PRompt
                je pc_prompt
                cmp ax,'id'                     ; DIsplay
                je pc_display
                cmp al,'f'                      ; F-key
                je pc_fkey
                jmp parse_config
pc_default:     mov di,offset default_cmd       ; "default" command
                call getline
                mov si,offset auto_cmd          ; add "auto"+null
                mov cx,5
                rep movsb
                jmp parse_config
pc_append:      push es                         ; "append" command
                push real_mode_seg
                pop es
                mov di,CmdlinePtr
                call getline
                mov CmdlinePtr,di
                pop es
                jmp parse_config
pc_timeout:     call getint                     ; "timeout" command
                jc parse_config
                push dx                         ; There are approx 1.D215h
                mov ax,0D215h                   ; clock ticks per 1/10 s
                mul dx
                pop ax
                add ax,dx
                mov KbdTimeOut,ax
                jmp parse_config
pc_display:     mov di,offset trackbuf
                push di
                call getline                    ; Get filename to display
                pop si
                mov di,offset MNameBuf          ; Mangled name buffer
                push di
                call mangle_name                ; Mangle file name
                pop di
                call searchdir                  ; Search for file
                jz parse_config                 ; File not found?
                call get_msg_file               ; Load and display file
                jmp parse_config
pc_prompt:      call getint                     ; "prompt" command
                jc parse_config
                mov ForcePrompt,dx
                jmp parse_config
pc_fkey:        sub ah,'1'
                jnb pc_fkey1
                mov ah,9                        ; F10
pc_fkey1:       xor cx,cx
                mov cl,ah
                push cx
                mov ax,1
                shl ax,cl
                or FKeyMap, ax                  ; Mark that we have this loaded
                mov di,offset trackbuf
                push di
                call getline                    ; Get filename to display
                pop si
                pop di
                shl di,4                        ; Multiply number by 16
                add di,offset FKeyName
                call mangle_name                ; Mangle file name
                jmp parse_config
no_config_file:
;
; Check whether or not we are supposed to display the boot prompt.
;
check_for_key:
                cmp ForcePrompt,0               ; Force prompt?
                jnz enter_command
                test KbdFlags,5Bh               ; Caps, Scroll, Shift, Alt
		jz auto_boot                    ; If neither, default boot

enter_command:
		mov si,offset boot_prompt
		call writestr

		mov di,offset command_line
;
; get the very first character -- we can either time
; out, or receive a character press at this time.  Some dorky BIOSes stuff
; a return in the buffer on bootup, so wipe the keyboard buffer first.
;
clear_buffer:   mov ah,1                        ; Check for pending char
                int 16h
                jz get_char_time
                xor ax,ax                       ; Get char
                int 16h
                jmp clear_buffer
get_char_time:  mov cx,KbdTimeOut
                and cx,cx
                jz get_char                     ; Timeout == 0 -> no timeout
                inc cx                          ; The first loop will happen
                                                ; immediately as we don't
                                                ; know the appropriate DX value
time_loop:      push cx
tick_loop:      push dx
                mov ah,1                        ; Check for pending keystroke
                int 16h
                jnz get_char_pop
                xor ax,ax
                int 1Ah                         ; Get time "of day"
                pop ax
                cmp dx,ax                       ; Has the timer advanced?
                je tick_loop
                pop cx
                loop time_loop                  ; If so, decrement counter
                jmp command_done                ; Timeout!
get_char_pop:   pop eax                         ; Clear the stack
get_char:       xor ax,ax                       ; Get char
		int 16h
                and al,al
                jz func_key
                cmp al,' '                      ; ASCII?
		jb not_ascii
		ja enter_char
		cmp di,offset command_line      ; Space must not be first
		je get_char
enter_char:     cmp di,max_cmd_len+offset command_line ; Check there's space
		jnb get_char
		stosb                           ; Save it
		call writechr                   ; Echo to screen
		jmp get_char
not_ascii:      cmp al,0Dh                      ; Enter
		je command_done
		cmp al,08h                      ; Backspace
		jne get_char
		cmp di,offset command_line      ; Make sure there is anything
		je get_char                     ; to erase
		dec di                          ; Unstore one character
		mov si,offset wipe_char         ; and erase it from the screen
		call writestr
		jmp get_char
func_key:
                push di
                cmp ah,68                       ; F10
                ja get_char
                sub ah,59                       ; F1
                jb get_char
                mov cl,ah
                shr ax,4                        ; Convert to x16
                mov bx,1
                shl bx,cl
                and bx,FKeyMap
                jz get_char                     ; Undefined F-key
                mov di,ax
                add di,offset FKeyName
                call searchdir
                jz fk_nofile
                call get_msg_file
                jmp fk_wrcmd
fk_nofile:
                mov si,offset crlf
                call writestr
fk_wrcmd:
                mov si,offset boot_prompt
                call writestr
                pop di                          ; Command line write pointer
                push di
                mov byte ptr [di],0             ; Null-terminate command line
                mov si,offset command_line
                call writestr                   ; Write command line so far
                pop di
                jmp get_char
auto_boot:
		mov si,offset default_cmd
		mov di,offset command_line
                mov cx,(max_cmd_len+4) shr 2
                rep movsd
		jmp load_kernel
command_done:
		mov si,offset crlf
		call writestr
		cmp di,offset command_line      ; Did we just hit return?
		je auto_boot
		xor al,al                       ; Store a final null
		stosb

load_kernel     label near                      ; Load the kernel now
;
; First we need to mangle the kernel name the way DOS would...
;
		mov si,offset command_line
		mov di,offset kernel_name
                push di
                call mangle_name
                pop di
		call searchdir
                jnz kernel_good
		mov si,offset err_notfound      ; Complain about missing kernel
		call writestr
bad_kernel:     mov si,offset kernel_name
		call writestr
		jmp enter_command               ; Ask user for clue
;
; kernel_corrupt: Called if the kernel file does not seem healthy
;
kernel_corrupt: push cs                         ; ES might be messed up
		pop es
		mov si,offset err_notkernel
		call writestr
		jmp bad_kernel

kernel_good:
;
; This is it!  We have a name (and location on the disk)... let's load
; that sucker!!
;
; A Linux kernel consists of three parts: boot sector, setup code, and
; kernel code.  The boot sector is never executed when using an external
; booting utility, but it contains some status bytes that are necessary.
; The boot sector and setup code together form exactly 5 sectors that
; should be loaded at 9000:0.  The subsequent code should be loaded
; at 1000:0.  For simplicity, we load the whole thing at 0F60:0, and
; copy the latter stuff afterwards.
;
; NOTE: In the previous code I have avoided making any assumptions regarding
; the size of a sector, in case this concept ever gets extended to other
; media like CD-ROM (if a PC manufacturer ever comes out with a bootable
; CD-ROM, of course).  However, a "sector" when it comes to Linux booting
; stuff means 512 bytes *no matter what*, so here I am using that piece
; of knowledge.
;
load_it:
		cmp dx,8                        ; Max kernel size = 80000h+
		jb kernel_sane                  ; real_mode_secs*512
		ja kernel_corrupt
		cmp ax,real_mode_secs*512
		ja kernel_corrupt               ; Too long to be a kernel

kernel_sane:    push ax
                push dx
                push si
		mov si,offset loading_msg
		call writestr
		pop si
                pop dx
                pop ax
;
; Now transfer the rest down to low memory
;
		push load_here_seg
		pop es
		assume es:NOTHING

                xor bx,bx
		add ax,ClustSize
		adc dx,0
		sub ax,1
		sbb dx,0
		div ClustSize                   ; # of clusters to get for PM
		mov KernelClust,ax
;
; Now, if we transfer these straight, we'll hit 64K boundaries.  Hence we
; have to see if we're loading more than 64K, and if so, load it step by
; step.
;
		mov dx,1                        ; 10000h
		xor ax,ax
		div ClustSize
		mov ClustPerMoby,ax             ; Clusters/64K
load_loop:
                mov ah,1                        ; Check for pending keystroke
                int 16h
                jz loadmoby                     ; If no pending keystroke
                xor ax,ax                       ; Load pending keystroke
                int 16h
                cmp al,27                       ; <ESC> aborts (DOS geeks)
                je do_abort
                cmp al,3                        ; So does Ctrl-C (UNIX geeks)
                jne load_loop
do_abort:
                push ds                         ; Restore ES = DS
                pop es
                mov si,offset aborted_msg
                call writestr
                jmp enter_command
loadmoby:
                push si
		mov si,offset dot_msg
		call writestr
		pop si
		mov cx,KernelClust
		cmp cx,ClustPerMoby
		jna last_moby
		mov cx,ClustPerMoby
last_moby:
		sub KernelClust,cx
		xor bx,bx                       ; Load at zero
		call getfssec
		jc load_done
		cmp KernelClust,0
		jz load_done

		mov ax,es                       ; Advance to next moby
		add ax,1000h
		mov es,ax
                jmp load_loop
load_done:
		push si
		mov si,offset dot_msg
		call writestr
		pop si
;
; Now copy the setup code to high memory
;
		push load_here_seg
		pop fs
		push real_mode_seg
		pop es
		assume es:real_mode_seg
		xor si,si
		mov di,si
		mov cx,real_mode_secs*80h
		rep movs dword ptr es:[di],fs:[si]
;
; If the default root device is set to FLOPPY (0000h), change to
; /dev/fd0 (0200h)
;
		cmp kern_def_root,0
		jne root_not_floppy
		mov kern_def_root,0200h
root_not_floppy:
;
; Copy the command line (append options have already been copied)
;
		mov kern_cmd_magic,0A33Fh       ; Command line magic no
		mov kern_cmd_offset,offset cmd_line_here
                mov di,CmdlinePtr
		mov si,offset command_prefix
		mov cx,(kern_cmd_len+3) shr 2
		rep movsd
;
; Copy the disk table to high memory, then re-initialize the FDC
;
		mov si,offset floppy_table
		mov di,offset linux_fdctab
		mov cx,3                        ; 12 bytes
		push di
		rep movsd
		pop di
		cli
		mov [fdctab1],di                ; Save new floppy tab pos
		mov [fdctab2],es
		sti
		xor ax,ax
		xor dx,dx
		int 13h
;
; Linux wants the floppy motor shut off before starting the kernel,
; at least bootsect.S seems to imply so
;
kill_motor:
		mov dx,03F2h
		xor al,al
		out dx,al
;
; Now we're as close to be done as we can be and still use our normal
; routines, print a CRLF to end the row of dots
;
		mov si,offset crlf
		call writestr
;
; Set up segment registers and the Linux real-mode stack
;
		mov ax,real_mode_seg
		mov ds,ax
		mov fs,ax
		mov gs,ax
		cli
		mov ss,ax
		mov sp,offset linux_stack
		sti
;
; We're done... now RUN THAT KERNEL!!!!
;
		jmp far ptr setup_entry

		assume ds:_text, es:_text
;
; searchdir: Search the root directory for a pre-mangled filename in
;            DS:DI.  This routine is similar to the one in the boot
;            sector, but is a little less Draconian when it comes to
;            error handling, plus it reads the root directory in
;            larger chunks than a sector at a time (which is probably
;            a waste of coding effort, but I like to do things right).
;
;            NOTE: This file considers finding a zero-length file an
;            error.  This is so we don't have to deal with that special
;            case elsewhere in the program (most loops have the test
;            at the end).
;
;            If successful:
;               ZF clear
;               SI      = cluster # for the first cluster
;               DX:AX   = file length in bytes
;            If unsuccessful
;               ZF set
;
searchdir       proc near
		mov ax,bsRootDirEnts
		mov DirScanCtr,ax
		mov ax,RootDirSize
		mov DirBlocksLeft,ax
		mov ax,RootDir1
		mov dx,RootDir2
scan_group:
		mov bp,DirBlocksLeft
		and bp,bp
		jz dir_return
		cmp bp,BufSafeSec
		jna load_last
		mov bp,BufSafeSec
load_last:
		sub DirBlocksLeft,bp
		push ax
		push dx
		mov ax,bsBytesPerSec
		mul bp
		add ax,offset trackbuf-31
		mov EndofDirSec,ax      ; End of loaded
		pop dx
		pop ax
		push bp                 ; Save number of sectors
		push ax                 ; Save present location
		push dx
		push di                 ; Save name
		mov bx,offset trackbuf
		call getlinsec
		pop di
		pop dx
		pop ax
		pop bp
		mov si,offset trackbuf
dir_test_name:  cmp byte ptr [si],0     ; Directory high water mark
		je dir_return           ; Failed
		push di
		push si
		mov cx,11               ; Filename = 11 bytes
		repe cmpsb
		pop si
		pop di
		je dir_success
		add si,32
		dec DirScanCtr
		jz dir_return           ; Out of it...
		cmp si,EndofDirSec
		jb dir_test_name
		add ax,bp               ; Increment linear sector number
		adc dx,0
		jmp scan_group
dir_success:
		mov ax,[si+28]          ; Length of file
		mov dx,[si+30]
		mov si,[si+26]          ; Cluster pointer
		mov bx,ax
		or bx,dx                ; Sets ZF iff DX:AX is zero
dir_return:
		ret
searchdir       endp
;
; writechr:     Write a single character in AL to the screen without
;               mangling any registers
;
writechr        proc near
		pusha
		mov ah,0Eh
		mov bx,0007h            ; white text on this page
		int 10h
		popa
		ret
writechr        endp
;
; get_msg_file: Load a text file and write its contents to the screen
;               Is called with SI and DX:AX set by routine searchdir
;
get_msg_file    proc near
get_msg_chunk:  push ax                         ; DX:AX = length of file
		push dx
		mov bx,offset trackbuf
		mov cx,BufSafe
		call getfssec
		pop dx
		pop ax
		push si                         ; Save current cluster
		mov si,offset trackbuf
		mov cx,BufSafeBytes             ; No more than many bytes
print_msg_file: push ax
		push dx
		lodsb
		cmp al,1Ah                      ; ASCII EOF?
		je msg_done_pop
		mov bx,00007h
		mov ah,0Eh                      ; Write to screen as tty
		int 10h                         ; Write in tty mode
		pop dx
		pop ax
		sub ax,1
		sbb dx,0
		mov bx,ax
		or bx,dx
		jz msg_done
		loop print_msg_file
		pop si
		jmp get_msg_chunk
msg_done_pop:
		pop eax                         ; Garbage value, OK if clobbered
msg_done:
		pop si
		ret
get_msg_file    endp
;
; open,getc:    Load a file a character at a time for parsing in a manner
;               similar to the C library getc routine.  Only one simultaneous
;               use is supported.  Note: "open" trashes the trackbuf.
;
;               open:   Input:  mangled filename in DS:DI
;                       Output: ZF set on file not found or zero length
;
;               getc:   Output: CF set on end of file
;                               Character loaded in AL
;
open            proc near
                call searchdir
                jz open_return
                pushf
                mov FBytes1,ax
                mov FBytes2,dx
                add ax,ClustSize
                adc dx,0
                sub ax,1
                sbb dx,0
                div ClustSize
                mov FClust,ax           ; Number of clusters
                mov FNextClust,si       ; Cluster pointer
                mov ax,EndOfGetCBuf     ; Pointer at end of buffer ->
                mov FPtr,ax             ;  nothing loaded yet
                popf                    ; Restore no ZF
open_return:    ret
open            endp
;
getc            proc near
                mov ax,FBytes1
                or ax,FBytes2
                jz getc_end
                mov si,FPtr
                cmp si,EndOfGetCBuf
                jb getc_loaded
                ; Buffer empty -- load another set
                mov cx,FClust
                cmp cx,BufSafe
                jna getc_oksize
                mov cx,BufSafe
getc_oksize:    sub FClust,cx           ; Reduce remaining clusters
                mov si,FNextClust
                mov bx,getcbuf
                push bx
                push es                 ; ES may be != DS, save old ES
                push ds                 ; Trackbuf is in DS, not ES
                pop es
                call getfssec           ; Load a trackbuf full of data
                mov FNextClust,si       ; Store new next pointer
                pop es                  ; Restore ES
                pop si                  ; SI -> newly loaded data
getc_loaded:    lodsb                   ; Load a byte
                mov FPtr,si             ; Update next byte pointer
                dec FBytes              ; Update bytes left counter (CF = 1)
getc_end:       cmc                     ; Set CF = 1 on EOF, 0 if not
                ret
getc            endp
;
; ungetc:       Push a character (in AL) back into the getc buffer
;               Note: if more than one byte is pushed back, this may cause
;               bytes to be written below the getc buffer boundary.  If there
;               is a risk for this to occur, the getcbuf base address should
;               be moved up.
;
ungetc          proc near
                mov si,FPtr
                dec si
                mov [si],al
                mov FPtr,si
                inc FBytes
                ret
ungetc          endp
;
; skipspace:    Skip leading whitespace using "getc".  If we hit end-of-line
;               or end-of-file, return with carry set; ZF = true of EOF
;               ZF = false for EOLN; otherwise CF = ZF = 0.
;
;               Otherwise AL = first character after whitespace
;
skipspace       proc near
skipspace_loop: call getc
                jc skipspace_eof
                cmp al,1Ah                      ; DOS EOF
                je skipspace_eof
                cmp al,0Ah
                je skipspace_eoln
                cmp al,' '
                jbe skipspace_loop
                ret                             ; CF = ZF = 0
skipspace_eof:  cmp al,al                       ; Set ZF
                stc                             ; Set CF
                ret
skipspace_eoln: add al,0FFh                     ; Set CF, clear ZF
                ret
skipspace       endp
;
; getkeyword:   Get a keyword from the current "getc" file; only the two
;               first characters are considered significant.
;
;               Lines beginning with ASCII characters 33-47 are treated
;               as comments and ignored; other lines are checked for
;               validity by scanning through the keywd_table.
;
;               The keyword and subsequent whitespace is skipped.
;
;               On EOF, CF = 1; otherwise, CF = 0, AL:AH = lowercase char pair
;
getkeyword      proc near
gkw_find:       call skipspace
                jz gkw_eof              ; end of file
                jc gkw_find             ; end of line: try again
                cmp al,'0'
                jb gkw_skipline         ; skip comment line
                push ax
                call getc
                pop bx
                jc gkw_eof
                mov bh,al               ; Move character pair into BL:BH
                or bx,2020h             ; Lower-case it
                mov si,offset keywd_table
gkw_check:      lodsw
                and ax,ax
                jz gkw_badline          ; Bad keyword, write message
                cmp ax,bx
                jne gkw_check
                push ax
gkw_skiprest:
                call getc
                jc gkw_eof_pop
                cmp al,'0'
                ja gkw_skiprest
                call ungetc
                call skipspace
                jz gkw_eof_pop
                jc gkw_badline_pop
                call ungetc             ; Return character to buffer
                clc                     ; Successful return
gkw_eof_pop:    pop ax
gkw_eof:        ret                     ; CF = 1 on all EOF conditions
gkw_badline_pop: pop ax
gkw_badline:    mov si,offset err_badcfg
                call writestr
                jmp gkw_find
gkw_skipline:   cmp al,10               ; Scan for LF
                je gkw_find
                call getc
                jc gkw_eof
                jmp gkw_skipline
getkeyword      endp
;
; getint:       Get a decimal integer from the "getc" file
;               Return CF if error; otherwise return integer in DX
;
getint          proc near
                xor dx,dx
                push dx
                call skipspace
                jc getint_ret
                pop dx
                cmp al,'0'
                jb getint_ret           ; CF set!
                cmp al,'9'
                ja getint_err
                mov dl,al
                sub dl,'0'
gi_moredigits:  push dx
                call getc
                pop dx
                jc getint_okret
                cmp al,'0'
                jb gi_endnum
                cmp al,'9'
                ja gi_endnum
                and ax,000fh
                shl dx,1
                add ax,dx
                shl dx,2
                add dx,ax
                jmp gi_moredigits
gi_endnum:      call ungetc             ; Return character to file
getint_okret:   clc
getint_ret:     ret
getint_err:     stc
                ret
getint          endp
;
; getline:      Get a command line, converting control characters to spaces
;               and collapsing streches to one.
;               The line is terminated by ^J, ^Z or EOF and is written
;               to ES:DI.  On return, DI points to first char after string.
;               CF is set if we hit EOF.
;
getline         proc near
                call skipspace
                jc gl_ret               ; CF set!
                call ungetc
                xor dx,dx               ; We did not start with a space
gl_fillloop:    push dx
                push di
                call getc
                pop di
                pop dx
                jc gl_ret               ; CF set!
                cmp al,' '
                jna gl_ctrl
                xor dx,dx
gl_store:       stosb
                jmp gl_fillloop
gl_ctrl:        cmp al,10
                je gl_ret               ; CF clear!
                cmp al,26
                je gl_eof
                and dl,dl
                jnz gl_fillloop         ; Ignore multiple spaces
                mov al,' '              ; Ctrl -> space
                inc dx
                jmp gl_store
gl_eof:         stc
gl_ret:         pushf                   ; We want the last char to be space!
                and dl,dl
                jnz gl_xret
                mov al,' '
                stosb
gl_xret:        popf
                ret
getline         endp

                ifdef debug             ; This code for debugging only
;
; dumpregs:     Dumps the contents of all registers
;
		assume es:NOTHING, fs:NOTHING, gs:NOTHING
dumpregs        proc near               ; When calling, IP is on stack
                pushf                   ; Store flags
		pusha
		push ds
		push es
		push fs
		push gs
		push cs                 ; Set DS <- CS
		pop ds
                cld                     ; Clear direction flag
                mov si,offset crlf
                call writestr
                mov bx,sp
                add bx,26
		mov si,offset regnames
                mov cx,2                ; 2*7 registers to dump
dump_line:      push cx
                mov cx,7                ; 7 registers per line
dump_reg:       push cx
                mov cx,4                ; 4 characters/register name
wr_reg_name:    lodsb
		call writechr
		loop wr_reg_name
                mov ax,ss:[bx]
		dec bx
		dec bx
		call writehex
		pop cx
		loop dump_reg
                mov al,0Dh              ; <CR>
                call writechr
                mov al,0Ah              ; <LF>
                call writechr
                pop cx
		loop dump_line
                pop gs
                pop fs
                pop es
                pop ds
		popa                    ; Restore the remainder
                popf                    ; Restore flags
		ret
dumpregs        endp

regnames        db ' IP: FL: AX: CX: DX: BX: SP: BP: SI: DI: DS: ES: FS: GS:'

;
; writehex:     Writes a 16-bit hexadecimal number (in AX)
;
writehex        proc near
		push bx
		push cx
		mov cx,4                ; 4 numbers
write_hexdig:   xor bx,bx
		push cx
		mov cx,4                ; 4 bits/digit
xfer_digit:     shl ax,1
		rcl bx,1
		loop xfer_digit
		push ax
		mov ax,bx
		or al,'0'
		cmp al,'9'
		jna ok_digit
		add al,'A'-'0'-10
ok_digit:       call writechr
		pop ax
		pop cx
		loop write_hexdig
		pop cx
		pop bx
		ret
writehex        endp

debug_magic     dw 0D00Dh

		endif
;
; mangle_name: Mangle a DOS filename pointed to by DS:SI into a buffer pointed
;              to by ES:DI; ends on encountering any whitespace
;

mangle_name     proc near
                mov cx,11                       ; # of bytes to write
mn_loop:
                lodsb
                cmp al,' '                      ; If control or space, end
                jna mn_end
                cmp al,'.'                      ; Period -> space-fill
                je mn_is_period
                cmp al,'a'
                jb mn_not_lower
                cmp al,'z'
                ja mn_not_uslower
                sub al,020h
                jmp short mn_not_lower
mn_is_period:   mov al,' '                      ; We need to space-fill
mn_period_loop: cmp cx,3                        ; If <= 3 characters left
                jbe mn_loop                     ; Just ignore it
                stosb                           ; Otherwise, write a period
                loop mn_period_loop             ; Dec CX and (always) jump
mn_not_uslower: cmp al,ucase_low
                jb mn_not_lower
                cmp al,ucase_high
                ja mn_not_lower
                mov bx,(offset ucase_tab)-ucase_low
                xlatb
mn_not_lower:   stosb
                loop mn_loop                    ; Don't continue if too long
mn_end:
                mov al,' '                      ; Space-fill name
                rep stosb                       ; Doesn't do anything if CX=0
                ret                             ; Done
mangle_name     endp
;
; Upper-case table for extended characters; this is technically code page 865,
; but code page 437 users will probably not miss not being able to use the
; cent sign in kernel images too much :-)
;
; The table only covers the range 129 to 164; the rest we can deal with.
;
ucase_low       equ 129
ucase_high      equ 164
ucase_tab       db 154, 144, 'A', 142, 'A', 143, 128, 'EEEIII'
		db 142, 143, 144, 146, 146, 'O', 153, 'OUUY', 153, 154
		db 157, 156, 157, 158, 159, 'AIOU', 165
;
; Various initialized or semi-initialized variables
;
boot_prompt     db 'boot: ',0
wipe_char       db 08h, ' ', 08h, 0
err_notfound    db 'Could not find kernel image: ',0
err_notkernel   db 0Dh, 0Ah, 'Invalid or corrupt kernel image: ',0
err_not386      db 'It appears your computer uses an 80286 or lower CPU.'
		db 0Dh, 0Ah
		db 'You cannot run Linux unless you have an 80386 or higher CPU'
		db 0Dh, 0Ah
		db 'in your machine.  If you get this message in error, hold'
		db 0Dh, 0Ah
		db 'down the Ctrl key while booting, and I will take your'
		db 0Dh, 0Ah
		db 'word for it.', 0Dh, 0Ah, 0
err_noram       db 'It appears your computer has less than 608K of low ("DOS")'
		db 0Dh, 0Ah
		db 'RAM.  Linux needs at least this amount to boot.  If you get'
		db 0Dh, 0Ah
		db 'this message in error, hold down the Ctrl key while'
		db 0Dh, 0Ah
		db 'booting, and I will take your word for it.', 0Dh, 0Ah, 0
err_badcfg      db 'Unknown keyword in SYSLINUX.CFG file.', 0Dh, 0Ah, 0
loading_msg     db 'Loading.'
dot_msg         db '.', 0
aborted_msg     db ' aborted.', 0Dh, 0Ah, 0
syslinux_cfg    db 'SYSLINUXCFG'
                align 2
keywd_table     db 'ap' ; append
                db 'de' ; default
                db 'ti' ; timeout
                db 'di' ; display
                db 'pr' ; prompt
                db 'f1' ; F1
                db 'f2' ; F2
                db 'f3' ; F3
                db 'f4' ; F4
                db 'f5' ; F5
                db 'f6' ; F6
                db 'f7' ; F7
                db 'f8' ; F8
                db 'f9' ; F9
                db 'f0' ; F10
                dw 0
ForcePrompt     dw 0                    ; Force prompt
KbdTimeOut      dw 0                    ; Keyboard timeout (if any)
FKeyMap         dw 0                    ; Bitmap for F-keys loaded
CmdLinePtr      dw cmd_line_here        ; Command line advancing pointer
kernel_name     db 11 dup (' ')
		db 0Dh, 0Ah, 0
linuxauto_cmd   db 'linux '
auto_cmd        db 'auto',0
                align 4
command_prefix  db 'BOOT_IMAGE='
command_line    equ $
default_cmd     equ $+(max_cmd_len+2)
ldlinux_end     equ default_cmd+(max_cmd_len+1)
kern_cmd_len    equ ldlinux_end-command_prefix
ldlinux_len     equ ldlinux_end-ldlinux_magic

_text           ends
		end bogus               ; Bogus entrypoint for EXE2BIN
