/* $Id: reqs.c,v 1.81 1999/12/03 00:34:54 rjkaes Exp $
 *
 * This is where all the work in tinyproxy is actually done. Incoming
 * connections are added to the active list of connections and then the header
 * is processed to determine what is being requested. tinyproxy then connects
 * to the remote server and begins to relay the bytes back and forth between
 * the client and the remote server. Basically, we sit there and sling bytes
 * back and forth. Optionally, we can weed out headers we do not want to send,
 * and also add a header of our own.
 *
 * Copyright (C) 1998  Steven Young
 * Copyright (C) 1999  Robert James Kaes (rjkaes@flarenet.com)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#ifdef HAVE_CONFIG_H
#include <defines.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/uio.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sysexits.h>
#include <assert.h>

#include "config.h"
#include "tinyproxy.h"
#include "sock.h"
#include "utils.h"
#include "conns.h"
#include "log.h"
#include "reqs.h"
#include "buffer.h"

#ifdef XTINYPROXY

static void Add_XTinyproxy_Header(struct conn_s *connptr)
{
	char *header_line;
	char ipaddr[PEER_IP_LENGTH];
	int length;

	assert(connptr);

	if (!(header_line = xmalloc(sizeof(char) * 32)))
		return;

	length = sprintf(header_line, "X-Tinyproxy: %s\r\n",
			 getpeer_ip(connptr->client_fd, ipaddr));

	unshift_buffer(connptr->cbuffer, header_line, length);
}

#endif

/*
 * Parse a client HTTP request and then establish connection.
 */
static int clientreq(struct conn_s *connptr)
{
	char *inbuf;
	char *domain, *location, *version = NULL, *port;
	int length;
	int port_no;
	char *request;

	char peer_ipaddr[PEER_IP_LENGTH];
	char peer_string[PEER_STRING_LENGTH];

	int fd;
	int retv;

	assert(connptr);

	if ((retv = readline(connptr->client_fd, connptr->cbuffer, &inbuf)) <=
	    0) {
		return retv;
	}

	length = strlen(inbuf) - 1;
	if (inbuf[length] == '\n')
		inbuf[length--] = '\0';

	if (inbuf[length] == '\r')
		inbuf[length--] = '\0';

	/* Log the incoming connection */
	if (!config.restricted) {
		log("Connect: %s [%s]",
		    getpeer_string(connptr->client_fd, peer_string),
		    getpeer_ip(connptr->client_fd, peer_ipaddr));
		log("Request: %s", inbuf);
	}

	/* Check to see if this is a simple request */
	if (!xstrstr(inbuf, "HTTP/", length, FALSE))
		connptr->simple_req = TRUE;

	if (!(domain = strchr(inbuf, ' '))) {
		log("ERROR clientreq: Incomplete line (%s)", peer_ipaddr);
		httperr(connptr, 400, "Unrecognizable request");
		goto COMMON_EXIT;
	}

	*domain++ = '\0';

	if (!xstrstr(domain, "http://", 7, FALSE)) {
		log("ERROR clientreq: Invalid scheme (%s)", peer_ipaddr);
		httperr(connptr, 400, "Invalid Scheme");
		goto COMMON_EXIT;
	}
	domain += 7;

	if (!connptr->simple_req) {
		if (!(version = strchr(domain, ' '))) {
			log("ERROR clientreq: Invalid line (%s)", peer_ipaddr);
			httperr(connptr, 400, "Invalid Line");
			goto COMMON_EXIT;
		}

		*version++ = '\0';
	}

	if (!(location = strchr(domain, '/'))) {
		location = inbuf + length;
	} else {
		*location++ = '\0';
	}

	if (!(port = strchr(domain, ':'))) {
		port_no = 80;
	} else {
		*port++ = '\0';
		if (strlen(port) > 0)
			port_no = atoi(port);
		else
			port_no = 80;
	}

	if ((strlen(config.stathost) > 0) &&
	    xstrstr(domain, config.stathost, strlen(domain), FALSE) == domain) {
		showstats(connptr);
		goto COMMON_EXIT;
	}

	/* Build a new request from the first line of the header */
	if (!(request = xmalloc(length + 1))) {
		log("ERROR clientreq: cannot allocate buffer for request (%s)",
		    peer_ipaddr);
		httperr(connptr, 503, "Internal Error");
		goto COMMON_EXIT;
	}

	/* We need to set the version number WE support */
	sprintf(request, "%s /%s HTTP/1.0\r\n", inbuf, location);

	if ((fd = opensock(domain, port_no)) < 0) {
		safefree(request);
		httperr(connptr, 500,
			"Unable to connect to host (cannot create sock)");
		stats.num_badcons++;
		goto COMMON_EXIT;
	}

#ifdef XTINYPROXY
	/* Add a X-Tinyproxy header which contains our IP address */
	if (config.my_domain
	    && xstrstr(domain, config.my_domain, strlen(domain), FALSE)) {
		Add_XTinyproxy_Header(connptr);
	}
#endif

	connptr->server_fd = fd;
	connptr->type = WAITCONN;

	/* Add the rewritten request to the buffer */
	unshift_buffer(connptr->cbuffer, request, strlen(request));

	/*
	 * HACK HACK HACK: When we're sending a POST there is no restriction
	 * on the length of the header. If we don't let all the header lines
	 * through, the POST will not work. This _definitely_ needs to be
	 * fixed. - rjkaes
	 */
	if (xstrstr(inbuf, "POST", 4, FALSE)) {
		connptr->clientheader = TRUE;
	}

      COMMON_EXIT:
	safefree(inbuf);
	return 0;
}

/*
 * Finish the client request
 */
static int clientreq_finish(struct conn_s *connptr)
{
	int sockopt, len = sizeof(sockopt);

	assert(connptr);

	if (getsockopt
	    (connptr->server_fd, SOL_SOCKET, SO_ERROR, &sockopt, &len) < 0) {
		if (errno == EINPROGRESS)
			return 0;

		log("ERROR clientreq_finish: getsockopt error (%s)");
		return -1;
	}

	if (sockopt != 0) {
		if (sockopt == EINPROGRESS)
			return 0;

		log("ERROR clientreq_finish: could not create connection (%s)",
		    strerror(errno));
		return -1;
	}

	connptr->type = RELAYCONN;
	stats.num_opens++;
	return 0;
}

/*
 * Check to see if the line is allowed or not depending on the anonymous
 * headers which are to be allowed.
 */
static int anonheader(char *line)
{
	struct allowedhdr_s *allowedptr = allowedhdrs;

	assert(line);
	assert(allowedhdrs);

	if (!xstrstr(line, "GET", 3, FALSE)
	    || !xstrstr(line, "POST", 4, FALSE)
	    || !xstrstr(line, "HEAD", 4, FALSE))
		return 1;

	for (allowedptr = allowedhdrs; allowedptr;
	     allowedptr = allowedptr->next) {
		if (!strncasecmp
		    (line, allowedptr->hdrname, strlen(allowedptr->hdrname))) {
			return 1;
		}
	}
	return 0;
}

/*
 * Used to read in the lines from the header (client side) when we're doing
 * the anonymous header reduction.
 */
static int readanonconn(struct conn_s *connptr)
{
	char *line = NULL;
	int retv;

	assert(connptr);

	if ((retv = readline(connptr->client_fd, connptr->cbuffer, &line)) <=
	    0) {
		return retv;
	}

	if ((line[0] == '\n') || (strncmp(line, "\r\n", 2) == 0)) {
		connptr->clientheader = TRUE;
	} else if (!anonheader(line)) {
		safefree(line);
		return 0;
	}

	push_buffer(connptr->cbuffer, line, strlen(line));
	return 0;
}

/*
 * Read in the bytes from the socket
 */
static int readconn(int fd, struct buffer_s *buffptr)
{
	int bytesin;

	assert(fd >= 0);
	assert(buffptr);

	if ((bytesin = readbuff(fd, buffptr)) < 0) {
		return -1;
	}
#ifdef __DEBUG__
	log("readconn [%d]: %d", fd, bytesin);
#endif

	stats.num_rx += bytesin;
	return bytesin;
}

/*
 * Write the bytes from the buffer to the socket
 */
static int writeconn(int fd, struct buffer_s *buffptr)
{
	int bytessent;

	assert(fd >= 0);
	assert(buffptr);

	if ((bytessent = writebuff(fd, buffptr)) < 0) {
		return -1;
	}

	stats.num_tx += bytessent;
	return bytessent;
}

/*
 * Factored out the common function to read from the client. It was used in
 * two different places with no change, so no point in having the same code
 * twice.
 */
static int read_from_client(struct conn_s *connptr, fd_set * readfds)
{
	assert(connptr);
	assert(readfds);

	if (FD_ISSET(connptr->client_fd, readfds)) {
		if (config.anonymous && !connptr->clientheader) {
			if (readanonconn(connptr) < 0) {
				shutdown(connptr->client_fd, 2);
				shutdown(connptr->server_fd, 2);
				connptr->type = FINISHCONN;
				return -1;
			}
		} else if (readconn(connptr->client_fd, connptr->cbuffer) < 0) {
			shutdown(connptr->client_fd, 2);
			shutdown(connptr->server_fd, 2);
			connptr->type = FINISHCONN;
			return -1;
		}
		connptr->actiontime = time(NULL);
	}

	return 0;
}

/*
 * Factored out the common write to client function since, again, it was used
 * in two different places with no changes.
 */
static int write_to_client(struct conn_s *connptr, fd_set * writefds)
{
	assert(connptr);
	assert(writefds);

	if (FD_ISSET(connptr->client_fd, writefds)) {
		if (writeconn(connptr->client_fd, connptr->sbuffer) < 0) {
			shutdown(connptr->client_fd, 2);
			shutdown(connptr->server_fd, 2);
			connptr->type = FINISHCONN;
			return -1;
		}
		connptr->actiontime = time(NULL);
	}

	return 0;
}

/*
 * All of the *_req functions handle the various stages a connection can go
 * through. I moved them out from getreqs because they handle a lot of error
 * code and it was indenting too far in. As you can see they are very simple,
 * and are only called once from getreqs, hence the inlining.
 */

inline static void newconn_req(struct conn_s *connptr, fd_set * readfds)
{
	assert(connptr);
	assert(readfds);

	if (FD_ISSET(connptr->client_fd, readfds)) {
		if (clientreq(connptr) < 0) {
			shutdown(connptr->client_fd, 2);
			connptr->type = FINISHCONN;
			return;
		}
		connptr->actiontime = time(NULL);
	}
}

inline static void waitconn_req(struct conn_s *connptr, fd_set * readfds,
				fd_set * writefds)
{
	assert(connptr);
	assert(readfds);
	assert(writefds);

	if (read_from_client(connptr, readfds) < 0)
		return;

	if (FD_ISSET(connptr->server_fd, readfds)
	    || FD_ISSET(connptr->server_fd, writefds)) {
		if (clientreq_finish(connptr) < 0) {
			shutdown(connptr->server_fd, 2);
			shutdown(connptr->client_fd, 2);
			connptr->type = FINISHCONN;
			return;
		}
		connptr->actiontime = time(NULL);
	}
}

inline static void relayconn_req(struct conn_s *connptr, fd_set * readfds,
				 fd_set * writefds)
{
	assert(connptr);
	assert(readfds);
	assert(writefds);

	if (read_from_client(connptr, readfds) < 0)
		return;

	if (FD_ISSET(connptr->server_fd, readfds)) {
		if (connptr->serverheader) {
			if (readconn(connptr->server_fd, connptr->sbuffer) < 0) {
				shutdown(connptr->server_fd, 2);
				connptr->type = CLOSINGCONN;
				return;
			}
		} else {
			/*
			 * We need to read in the first line to rewrite the
			 * version back down to HTTP/1.0 (if needed)
			 */
			char *line = NULL, *ptr, *newline;
			int retv;

			if (
			    (retv =
			     readline(connptr->server_fd, connptr->sbuffer,
				      &line)) < 0) {
				shutdown(connptr->server_fd, 2);
				httperr(connptr, 500, "Server Closed Early");
				return;
			} else if (retv == 0)
				return;

			connptr->serverheader = TRUE;

			if (strncasecmp(line, "HTTP/1.0", 8)) {
				/* Okay, we need to rewrite it then */
				if (!(ptr = strchr(line, ' '))) {
					shutdown(connptr->server_fd, 2);
					httperr(connptr, 500,
						"There was Server Error");
					return;
				}
				ptr++;

				if (!(newline = xmalloc(strlen(line) + 1))) {
					shutdown(connptr->server_fd, 2);
					httperr(connptr, 503,
						"No Memory Available");
					return;
				}

				sprintf(newline, "HTTP/1.0 %s", ptr);
				safefree(line);
				line = newline;
			}

			push_buffer(connptr->sbuffer, line, strlen(line));
		}
		connptr->actiontime = time(NULL);
	}

	if (write_to_client(connptr, writefds) < 0)
		return;

	if (FD_ISSET(connptr->server_fd, writefds)) {
		if (writeconn(connptr->server_fd, connptr->cbuffer) < 0) {
			shutdown(connptr->server_fd, 2);
			connptr->type = CLOSINGCONN;
			return;
		}
		connptr->actiontime = time(NULL);
	}
}

inline static void closingconn_req(struct conn_s *connptr, fd_set * writefds)
{
	assert(connptr);
	assert(writefds);

	write_to_client(connptr, writefds);
}

/*
 * Check against the valid subnet to see if we should allow the access
 */
static int validuser(int fd)
{
	char ipaddr[PEER_IP_LENGTH];

	assert(fd >= 0);

	if (config.subnet == NULL)
		return 1;

	if (!strncmp(config.subnet, getpeer_ip(fd, ipaddr),
		     strlen(config.subnet))) {
		return 1;
	} else {
		return 0;
	}
}

/*
 * Loop that checks for new connections, dispatches to the correct
 * routines if bytes are pending, checks to see if it's time for a
 * garbage collect.
 */
int getreqs(void)
{
	static unsigned int garbc = 0;
	fd_set readfds, writefds;
	struct conn_s *connptr;
	int fd;
	struct timeval tv;

	char peer_ipaddr[PEER_IP_LENGTH];
	char peer_string[PEER_STRING_LENGTH];

	if (setup_fd < 0) {
		log("ERROR getreqs: setup_fd not a socket");
		return -1;
	}

	/* Garbage collect the dead connections and close any idle ones */
	if (garbc++ >= GARBCOLL_INTERVAL) {
		garbcoll();
		garbc = 0;
	}
	conncoll();

	FD_ZERO(&readfds);
	FD_ZERO(&writefds);
	FD_SET(setup_fd, &readfds);

	for (connptr = connections; connptr; connptr = connptr->next) {
#ifdef __DEBUG__
		log("Connptr: %p - %d / client %d server %d", connptr,
		    connptr->type, connptr->client_fd, connptr->server_fd);
#endif
		switch (connptr->type) {
		case NEWCONN:
			if (buffer_size(connptr->cbuffer) < MAXBUFFSIZE)
				FD_SET(connptr->client_fd, &readfds);
			else {
				httperr(connptr, 414,
					"Your Request is way too long.");
			}
			break;

		case WAITCONN:
			FD_SET(connptr->server_fd, &readfds);
			FD_SET(connptr->server_fd, &writefds);

			if (buffer_size(connptr->cbuffer) < MAXBUFFSIZE)
				FD_SET(connptr->client_fd, &readfds);
			break;

		case RELAYCONN:
			if (buffer_size(connptr->sbuffer) > 0)
				FD_SET(connptr->client_fd, &writefds);
			if (buffer_size(connptr->sbuffer) < MAXBUFFSIZE)
				FD_SET(connptr->server_fd, &readfds);

			if (buffer_size(connptr->cbuffer) > 0)
				FD_SET(connptr->server_fd, &writefds);
			if (buffer_size(connptr->cbuffer) < MAXBUFFSIZE)
				FD_SET(connptr->client_fd, &readfds);

			break;

		case CLOSINGCONN:
			if (buffer_size(connptr->sbuffer) > 0)
				FD_SET(connptr->client_fd, &writefds);
			else {
				shutdown(connptr->client_fd, 2);
				shutdown(connptr->server_fd, 2);
				connptr->type = FINISHCONN;
			}
			break;

		default:
			break;
		}
	}

	/* Set a 60 second time out */
	tv.tv_sec = 60;
	tv.tv_usec = 0;

	if (select(FD_SETSIZE, &readfds, &writefds, NULL, &tv) <= 0) {
#ifdef __DEBUG__
		log("select error: %s", strerror(errno));
#endif
		return 0;
	}

	/* Check to see if there are new connections pending */
	if (FD_ISSET(setup_fd, &readfds) && (fd = listen_sock()) >= 0) {
		new_conn(fd);	/* make a connection from the FD */

		if (validuser(fd)) {
			if (config.cutoffload && (load > config.cutoffload)) {
				stats.num_refused++;
				httperr(connptr, 503,
					"tinyproxy is not accepting connections due to high system load");
			}
		} else {
			httperr(connptr, 403,
				"You are not authorized to use the service.");
			log("AUTH Rejected connection from %s [%s]",
			    getpeer_string(fd, peer_string),
			    getpeer_ip(fd, peer_ipaddr));
		}
	}

	/* 
	 * Loop through the connections and dispatch them to the appropriate
	 * handler
	 */
	for (connptr = connections; connptr; connptr = connptr->next) {
		switch (connptr->type) {
		case NEWCONN:
			newconn_req(connptr, &readfds);
			break;

		case WAITCONN:
			waitconn_req(connptr, &readfds, &writefds);
			break;

		case RELAYCONN:
			relayconn_req(connptr, &readfds, &writefds);
			break;

		case CLOSINGCONN:
			closingconn_req(connptr, &writefds);
			break;

		default:
			break;
		}
	}

	return 0;
}
