#ifndef VISUAL_SPHMODEL_H
#define VISUAL_SPHMODEL_H

// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.

#include <cmath>
#include "vertex.h"

// sphmodel.h - Deep magic to generate sphere models
//   I generate spheres by subdividing an icosahedron, a Platonic
//     solid with 12 vertices and 20 sides, each an equilateral
//     triangle.  Each triangle is subdivided into four, yielding
//     a figure with 80 sides, 320 sides, and so on.  The complexity
//     is due to the fact that an 80-face "sphere" should have 42
//     vertices, but a naiive algorithm will generate many more
//     (multiple vertices at the same spot).  I avoid this with lots
//     of unmaintainable pointer goop.  If you have to modify this
//     file significantly, just start over.  I warned you!


namespace visual {

struct sph_model
{
	float* verts;
	vertex* proj;
	float* color;
	int* indices;
	int nverts;
	int ni;

	static inline void
	normalize(float v[3])
	{
		float n = v[0]*v[0] + v[1]*v[1] + v[2]*v[2];
		float m = 1.0 / std::sqrt( n );

		v[0] *= m;
		v[1] *= m;
		v[2] *= m;
	}

	static sph_model& get( int depth);

	inline float*
	newe( int span)
	{
		float* e = verts + 3*(nverts - 1);  // one before the beginning
		nverts += span-1;
		return e;
	}

	sph_model( int depth);

	inline float*
	avgptr( float* a, float* b)
	{ return (float*)( int(a) + ((int(b)-int(a))>>1) ); }

	void subdivide( int span, float* v1, float* v2, float *v3
		, float* s1, float* s2, float* s3
		, float* e1, float* e2, float* e3 );
};

} // !namespace visual

#endif // !VISUAL_SPHMODEL_H
