#############################################################################
# Copyright (C) DSTC Pty Ltd (ACN 052 372 577) 1993, 1994, 1995.
# Unpublished work.  All Rights Reserved.
#
# The software contained on this media is the property of the
# DSTC Pty Ltd.  Use of this software is strictly in accordance
# with the license agreement in the accompanying LICENSE.DOC file.
# If your distribution of this software does not contain a
# LICENSE.DOC file then you have no rights to use this software
# in any manner and should contact DSTC at the address below
# to determine an appropriate licensing arrangement.
# 
#      DSTC Pty Ltd
#      Level 7, Gehrmann Labs
#      University of Queensland
#      St Lucia, 4072
#      Australia
#      Tel: +61 7 3365 4310
#      Fax: +61 7 3365 4311
#      Email: enquiries@dstc.edu.au
# 
# This software is being provided "AS IS" without warranty of
# any kind.  In no event shall DSTC Pty Ltd be liable for
# damage of any kind arising out of or in connection with
# the use or performance of this software.
#
# Project:  Python Tools
#
# File:     ast_doc.py
#
# Description:
#           Functions to extract documentation from Python ASTs.
#
# History:
#           19-Dec-1995  Martin Chilvers (martin) : created
#
# "@(#)$RCSfile: ast_doc.py,v $ $Revision: 1 $"
#
#$Log: /Gendoc/ast_doc.py $
# 
# 1     98-04-01 13:15 Daniel
# Revision 1.2  1996/07/08  05:33:48  omfadmin
# Incorporated Fred Drake's changes (Python 1.4 support).
#
# Revision 1.1  1996/06/13  17:57:56  omfadmin
# Initial revision
#
#Revision 1.4  1995/12/19 01:43:24  martin
#Added DSTC header.
#
#
#############################################################################

""" Functions to extract documentation from Python ASTs. 

The ASTs must be in tuple form as produced by the function 'ast2tuple' in the
parser module.

The functions intended for public consumption are:-

    'module_doc'    return the documentation for a module.
    'class_doc'     return the documentation for a class.
    'function_doc'  return the documentation for a function.

All other functions are intended for private use only (but you can use them
if you *really* want to ;^)

"""

# Built-in/standard modules.
import string

# Local modules.
import ast
import token
import symbol
import sys

# The pattern key used to match documentation strings.
DOC_PATTERN = ast.AST_PATTERN + 'doc'

#  This allows us to work with either Python 1.3 or 1.4.
_py_version = string.split(sys.version)[0][:3]
if _py_version == "1.3":
    _inner = (symbol.atom, (token.STRING, DOC_PATTERN))
else:
    _inner = (symbol.power, (symbol.atom, (token.STRING, DOC_PATTERN)))

# The structure of a documentation node!
DOC_NODE = (symbol.stmt,
	    (symbol.simple_stmt,
	     (symbol.small_stmt,
	      (symbol.expr_stmt,
	       (symbol.testlist,
		(symbol.test,
		 (symbol.and_test,
		  (symbol.not_test,
		   (symbol.comparison,
		    (symbol.expr,
		     (symbol.xor_expr,
		      (symbol.and_expr,
		       (symbol.shift_expr,
			(symbol.arith_expr,
			 (symbol.term,
			  (symbol.factor,
			   _inner)))))))))))))),
	     (token.NEWLINE, '')))

del _py_version, _inner

def module_doc(node):
    """Return the documentation for a module.

    The documentation for a module is a single documentation string.

    """

    return node_doc(node)

def import_name(node):
    """Return the name of an imported module."""

    dotted_name = ast.search(node, [symbol.dotted_name], [], 1)[0]

    # dotted_name is a tuple in the form:
    #  (symbol.dotted_name, (1, mod_pack) [, (23, '.'), (1, mod_pack) ...])
    # where mod_pack is a string containing either a package or a
    # module name. (the '[' and ']' characters denote optional tuples, *not*
    # a Python list!

    # 'cat' concatenates the names and the dots.
    cat = lambda str, elem: str+elem[1]
    return reduce(cat, dotted_name[1:], '')


def class_doc(node):
    """ Return the documentation for a class.

    The documentation for a class is a tuple of the following form:-

    (ClassName, [BaseClasses], Documentation)

    """

    # Get the name of the class and any associated documentation (the class
    # suite is the last element in the node).
    classname = node[2][1]
    documentation = node_doc(node[-1])

    # Now get the classes base classes (to make the search quicker, only look
    # at the class header not its associated suite).
    header = node[:-1]
    lst_lists = ast.search(header, [symbol.testlist], [], 1)

    if len(lst_lists) > 0:
	lst_factors = ast.search(lst_lists[0], [symbol.factor], [])
	inheritance = map(ast.str, lst_factors)
    else:
	inheritance = []

    return (classname, inheritance, documentation)


def function_doc(node):
    """ Return the documentation for a function.

    The documentation for a function is a tuple of the following form:-

    (FunctionName, [(Parameter, DefaultValue)], Documentation)

    """

    # Get the name of the function and any associated documentation (the
    # function suite is the last element in the node).
    functionname = node[2][1]
    documentation = node_doc(node[-1])

    # Now get the functions parameters and their default values (to make the
    # search quicker, only look at the function header not its associated
    # suite!).
    header = node[:-1]
    parameters = []

    # Find the parameter list (if there is one!).
    lst_params = ast.search(header, [symbol.varargslist], [], 1)
    if len(lst_params) > 0:
	node_body = ast.node_body(lst_params[0])

	# Scan the parameter list for formal parameter nodes.
	for i in range (len(node_body)):

	    # If we have found a formal parameter node.
	    if ast.node_type(node_body[i]) == symbol.fpdef:

		# Get the parameter name.
		name = ast.str(node_body[i])

		# Get the parameter's default value (the parameter has a
		# default value if the next token is an equals sign!).
		if (i < len(node_body)-1		
		    and ast.node_type(node_body[i+1]) == token.EQUAL):
		    default = ast.str(node_body[i+2])
		else:
		    default = None

		parameters.append((name, default))

	    # The function takes a variable number of arguments.
	    elif ast.node_type(node_body[i]) == token.STAR:
		name = '*' + node_body[i+1][1]
		parameters.append((name, None))

    return (functionname, parameters, documentation)


def node_doc(node):
    """ Return the documentation string for a node (if there is one!).

    A node is considered to have documentation if the FIRST statment it
    contains is a simple string expression statement (ie. it matches the
    structure in DOC_NODE).

    """
    lst_stmts = ast.search(node, [symbol.stmt], [], 1)
    if len(lst_stmts) > 0:
	d_matches = {}
	if ast.match(lst_stmts[0], DOC_NODE, d_matches):
	    documentation = d_matches[DOC_PATTERN]
	else:
	    documentation = None
    else:
	documentation = None

    return documentation
