// scaledarea.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// BorderedArea is a class that allowed me to set an arbitrary empty space
// (Margin) on all four sides of a Canvas, so that all drawing operations would
// be automatically offset to fit.  There is probably a better way!
//
// ScaledArea is a BorderedArea to which one or more Scale objects is attached.
// As the vertical and horizontal scaling factors of the ScaledArea's drawing
// region change, the Scales are updated accordingly.  ScaledArea is the base
// class for all Graph subclasses.

#ifndef SCALEDAREA_H
#ifdef __GNUG__
#pragma interface
#endif
#define SCALEDAREA_H

#include <InterViews/interactor.h>
#include "margin.h"
#include "range.h"

class Perspective;
class Brush;
class Painter;

class BorderedArea : public Interactor {
public:
	BorderedArea();
	virtual ~BorderedArea();
	// redefined InterViews methods
	redefined void Update();
	// new methods
	virtual int currentHeight();
	virtual int currentWidth();
	virtual void setVertMargins(int top, int bottom) {
		myMargin->top = top; myMargin->bottom = bottom; 
	}
	virtual void setHorizMargins(int left, int right) {
		myMargin->left = left; myMargin->right = right;
	}
	virtual int topBorder() { return myMargin->top; }
	virtual int bottomBorder() { return myMargin->bottom; }
	int leftEdge() { return myMargin->left; }
	int rightEdge() { return xmax - myMargin->right + 2; }
	Margin* margin() { return myMargin; }
	Painter *getOutput() { return output; }
protected:
	// new methods
	int widthChanged() { return xmax != oldxmax; }
	int heightChanged() { return ymax != oldymax; }
	virtual boolean IsMapped();
	virtual boolean isResized();
	void resetDimensions() { oldxmax = xmax; oldymax = ymax; }
	virtual void doAdjust() {}
private:
	Margin *myMargin;
	int oldxmax, oldymax;
	void Init();
};

// ScaledArea is a BorderedArea dependent on a VScale

class Scale;

class ScaledArea : public BorderedArea {
	typedef BorderedArea Super;
public:
	ScaledArea(const Range &range, Scale *vs=nil);
	virtual ~ScaledArea();
	// redefined InterViews methods
	redefined void Reconfig();
	redefined void Update();
	// new methods
	virtual int vertOriginLocation();
	virtual int horizOriginLocation() { return leftEdge(); }
	virtual double currentHGrain();
	virtual double currentVGrain();
	virtual Range currentVerticalRange() {
		return verticalRange.check();
	}
	virtual Range verticalScaleRange() {
		return currentVerticalRange();
	}
	Range getReferenceRange() { return referenceRange; }
	void setReferenceRange(const Range &r);
	void attachVScale(Scale *s) { vscale = s; }
	void removeVScale(Scale *s) { if(vscale == s) vscale = nil; }
	Scale *vScale() { return vscale; }
protected:
	// redefined InterViews methods
	redefined void Resize();

	redefined void doAdjust();
	// new methods
	void checkScales();
	virtual void adjustVerticalRange();
	virtual double vertScaleFactor();
protected:
	// new members
	Scale *vscale;			// pointers to scales (if any)
private:
	friend class SimplePlot;
	Range referenceRange;		// fixed range
	Range verticalRange;		// variable visible range
private:
	void Init();
};

#endif
