(**************************************************************************)
(*   Copyright (c) 2003 Mikhail Fedotov <mikhail@kittown.com>             *)
(*                                                                        *)
(*  Permission is hereby granted, free of charge, to any person           *)
(*  obtaining a copy of this software and associated documentation files  *)
(*  (the "Software"), to deal in the Software without restriction,        *)
(*  including without limitation the rights to use, copy, modify, merge,  *)
(*  publish, distribute, sublicense, and/or sell copies of the Software,  *)
(*  and to permit persons to whom the Software is furnished to do so,     *)
(*  subject to the following conditions:                                  *)
(*                                                                        *)
(*  The above copyright notice and this permission notice shall be        *)
(*  included in all copies or substantial portions of the Software.       *)
(*                                                                        *)
(*  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,       *)
(*  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES       *)
(*  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND              *)
(*  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS   *)
(*  BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN    *)
(*  ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN     *)
(*  CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE      *)
(*  SOFTWARE.                                                             *)
(**************************************************************************)

(* This file does some aggressive testing for the sqlite library. The main
    purpose is to provocate Ocaml GC to cause segmentation faults and other
    wonderful things in case there is still some memory-related error in the
    database stub code left. 
  
    No effort have been made to make this source look beautiful. :-)
*)

open Sqlite;;

let print_query_result vm =

  let print_strings desc sep strings =
    let num = Array.length strings in
    let () = print_string (desc^": ") in
    if num > 0 then
      begin
        print_string strings.(0);
        for i = 1 to num - 1 do
          print_string (sep^strings.(i))
        done;
        print_newline ()
      end
  in

  let print_colnames () =
    print_strings "Column names" "|" (column_names vm)
  in

  let print_string_options desc sep default_string string_options =
    let num = Array.length string_options in
    let print_string_option string_option =
      match string_option with
        None -> print_string default_string
      | Some s -> print_string s
    in
    let () = print_string (desc^": ") in
    if num > 0 then begin
      print_string_option string_options.(0);
      for i = 1 to num - 1 do
        print_string sep;
        print_string_option string_options.(i)
      done;
      print_newline ()
    end
  in

  let print_coltypes () =
    print_strings "Column types" "|" (column_types vm)
  in

  let print_values values =
    print_string_options "Column values" "|" "NULL" values
  in

  let rec print_rows () =
    let () = print_values (step_opt vm) in
    print_rows ()
  in

  try
    let values = step_opt vm in
    let () = print_colnames() in
    let () = print_coltypes () in
    let () = print_values(values) in
    try
      print_rows ()
    with Sqlite_done -> ()
  with Sqlite_done ->
    begin
      print_colnames ();
      print_coltypes ();
      print_string "Empty query result\n"
    end
;;

let do_this () = 

  let db = db_open "./test.db" in

  let () = print_string "Drop table tbl1 if exists (ignore the exception if it does not)\n" in
  let () = try 
    exec db "DROP TABLE tbl1;"
  with Sqlite_error s -> () in

  let () = print_string "Create table tbl1\n" in
  let () = try 
    exec db "CREATE TABLE tbl1(zero, one varchar(10), two smallint, three);"
  with Sqlite_error s -> raise (Failure "error while creating table") in

  let () = print_string "Check printing of column names & types when there are\n" in
  let () = print_string "zero rows in result so Sqlite_done must be raised immediately\n" in
  let (vm, _, _) = Sqlite.compile db "SELECT * FROM tbl1" 0 true in
  let () = print_query_result vm in

  let () = print_string "Fill the table with some data\n" in
  let () = exec db "INSERT INTO tbl1 values(0, '1', '2', '10');" in
  let () = exec db "INSERT INTO tbl1 values(0, '3', '4', NULL);" in


  let () = print_string "Last insert rowid = " in
  let () = print_int (last_insert_rowid db) in
  let () = print_newline () in


  let () = exec db "PRAGMA full_column_names = ON;" in
    
  let () = print_string "Print all table contents\n" in
  let (vm, l, _) = Sqlite.compile db "SELECT * FROM tbl1; This is the remainder of the string" 0 true in
  let () = print_query_result vm in

  let () = print_int l in
  let () = print_newline() in

  let () = print_string "Interrupt a query in the middle\n" in
  let (vm, _, _) = Sqlite.compile db "SELECT * FROM tbl1;" 0 true in
  let _ = step_opt vm in
  let () = finalize vm in
  let () = try 
    let _ = step_opt vm in ()
  with Sqlite_error s -> print_string "Sqlite misuse detected correctly\n" in

  let () = print_string "Vm finalized ok in the middle of the query\n" in

  let (vm, _, _) = compile db "select * from tbl1;" 0 true in
  let _ = step_simple vm in
  let values = step vm "FAKE NULL VALUE" in
  let () = for i = 0 to Array.length values - 1 do
      print_string values.(i);
      print_newline()
  done in
  let () = finalize vm in

  let (vm, _, bt) = compile db "select * from tbl1;" 0 true in
  let () = if (not bt) then (print_string "All ok\n") else raise (Failure("Error")) in
  let _ = step_simple vm in
  let () = try
    let values = step_simple vm in ()
  with Sqlite_null_value -> print_string "Sqlite_null_value works as expected\n" in
  let () = finalize vm in

  let (vm, iii, bt) = compile db "select * from tbl1;select 2;" 0 true in
  let () = if (bt) then (print_string "All ok\n") else raise (Failure("Error")) in
  let () = if (iii = 19) then (print_string "All ok\n") else raise (Failure("Error")) in
  let _ = step_simple vm in
  let () = try
    let values = step_simple vm in ()
  with Sqlite_null_value -> print_string "Sqlite_null_value works as expected\n" in
  let () = finalize vm in
  db_close db
;;

(* run this for a hundred times to provocate GC in case there are memory-related errors *)
for i = 1 to 100 do 
do_this () done;;
