// This file may be redistributed and modified only under the terms of
// the GNU Lesser General Public License (See COPYING for details).
// Copyright (C) 2000-2001 Stefanus Du Toit and Alistair Riddoch

#ifndef ATLAS_OBJECTS_ROOT_H
#define ATLAS_OBJECTS_ROOT_H

#include <Atlas/Message/Encoder.h>
#include <Atlas/Message/Element.h>

namespace Atlas { namespace Objects {

/** An exception indicating the requested attribute does not exist.
 *
 * This is thrown by Root::getAttr() [and derivatives thereof!]
 */
class NoSuchAttrException
{
public:
    NoSuchAttrException(const std::string& name) : name(name) {}
    std::string name;
};

/** All objects inherit from this


  
*/
class Root {
public:
    /// Construct a Root instance.
    Root();
    /// Construct a Root class definition.
    Root(const char * id);
protected:
    /// Constructor for classes which inherit from this one
    Root(const char * id, const char * parent);
public:
    /// Default destructor.
    virtual ~Root();

    /// Create a new class for Root.
    static Root Class();

    /// Check whether the attribute "name" exists.
    virtual bool hasAttr(const std::string& name) const;
    /// Retrieve the attribute "name". Throws NoSuchAttrException if it does
    /// not exist.
    virtual Atlas::Message::Element getAttr(const std::string& name)
        const throw (NoSuchAttrException);
    /// Set the attribute "name" to the value given by "attr".
    virtual void setAttr(const std::string& name,
                         const Atlas::Message::Element& attr);
    /// Remove the attribute "name". This will not work for static attributes.
    virtual void removeAttr(const std::string& name);

    /// Set the "parents" attribute.
    inline void setParents(const Atlas::Message::Element::ListType& val);
    /// Set the "id" attribute.
    inline void setId(const std::string& val);
    /// Set the "objtype" attribute.
    inline void setObjtype(const std::string& val);
    /// Set the "name" attribute.
    inline void setName(const std::string& val);

    /// Retrieve the "parents" attribute.
    inline const Atlas::Message::Element::ListType& getParents() const;
    inline Atlas::Message::Element::ListType& getParents();
    /// Retrieve the "id" attribute.
    inline const std::string& getId() const;
    inline std::string& getId();
    /// Retrieve the "objtype" attribute.
    inline const std::string& getObjtype() const;
    inline std::string& getObjtype();
    /// Retrieve the "name" attribute.
    inline const std::string& getName() const;
    inline std::string& getName();
    
    /// Convert this object to a Message::Element.
    virtual Atlas::Message::Element asObject() const;
    /// Convert this object to an std::map.
    virtual Atlas::Message::Element::MapType asMap() const;

    /// Send the contents of this object to a Bridge.
    virtual void sendContents(Atlas::Bridge* b) const;

protected:
    std::map<std::string, Atlas::Message::Element> attributes;
    Atlas::Message::Element::ListType attr_parents;
    std::string attr_id;
    std::string attr_objtype;
    std::string attr_name;

    inline void sendParents(Atlas::Bridge*) const;
    inline void sendId(Atlas::Bridge*) const;
    inline void sendObjtype(Atlas::Bridge*) const;
    inline void sendName(Atlas::Bridge*) const;
};

// 
// Inlined member functions follow.
//

void Root::setParents(const Atlas::Message::Element::ListType& val)
{
    attr_parents = val;
}

void Root::setId(const std::string& val)
{
    attr_id = val;
}

void Root::setObjtype(const std::string& val)
{
    attr_objtype = val;
}

void Root::setName(const std::string& val)
{
    attr_name = val;
}

const Atlas::Message::Element::ListType& Root::getParents() const
{
    return attr_parents;
}

Atlas::Message::Element::ListType& Root::getParents()
{
    return attr_parents;
}

const std::string& Root::getId() const
{
    return attr_id;
}

std::string& Root::getId()
{
    return attr_id;
}

const std::string& Root::getObjtype() const
{
    return attr_objtype;
}

std::string& Root::getObjtype()
{
    return attr_objtype;
}

const std::string& Root::getName() const
{
    return attr_name;
}

std::string& Root::getName()
{
    return attr_name;
}

} }

#endif
