﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/securityhub/SecurityHub_EXPORTS.h>
#include <aws/securityhub/model/RuleStatus.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace SecurityHub {
namespace Model {

/**
 * <p> Metadata for automation rules in the calling account. The response includes
 * rules with a <code>RuleStatus</code> of <code>ENABLED</code> and
 * <code>DISABLED</code>. </p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/AutomationRulesMetadata">AWS
 * API Reference</a></p>
 */
class AutomationRulesMetadata {
 public:
  AWS_SECURITYHUB_API AutomationRulesMetadata() = default;
  AWS_SECURITYHUB_API AutomationRulesMetadata(Aws::Utils::Json::JsonView jsonValue);
  AWS_SECURITYHUB_API AutomationRulesMetadata& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_SECURITYHUB_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p> The Amazon Resource Name (ARN) for the rule. </p>
   */
  inline const Aws::String& GetRuleArn() const { return m_ruleArn; }
  inline bool RuleArnHasBeenSet() const { return m_ruleArnHasBeenSet; }
  template <typename RuleArnT = Aws::String>
  void SetRuleArn(RuleArnT&& value) {
    m_ruleArnHasBeenSet = true;
    m_ruleArn = std::forward<RuleArnT>(value);
  }
  template <typename RuleArnT = Aws::String>
  AutomationRulesMetadata& WithRuleArn(RuleArnT&& value) {
    SetRuleArn(std::forward<RuleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Whether the rule is active after it is created. If this parameter is equal
   * to <code>ENABLED</code>, Security Hub starts applying the rule to findings and
   * finding updates after the rule is created. To change the value of this parameter
   * after creating a rule, use <a
   * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateAutomationRules.html">
   * <code>BatchUpdateAutomationRules</code> </a>. </p>
   */
  inline RuleStatus GetRuleStatus() const { return m_ruleStatus; }
  inline bool RuleStatusHasBeenSet() const { return m_ruleStatusHasBeenSet; }
  inline void SetRuleStatus(RuleStatus value) {
    m_ruleStatusHasBeenSet = true;
    m_ruleStatus = value;
  }
  inline AutomationRulesMetadata& WithRuleStatus(RuleStatus value) {
    SetRuleStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An integer ranging from 1 to 1000 that represents the order in which the rule
   * action is applied to findings. Security Hub applies rules with lower values for
   * this parameter first. </p>
   */
  inline int GetRuleOrder() const { return m_ruleOrder; }
  inline bool RuleOrderHasBeenSet() const { return m_ruleOrderHasBeenSet; }
  inline void SetRuleOrder(int value) {
    m_ruleOrderHasBeenSet = true;
    m_ruleOrder = value;
  }
  inline AutomationRulesMetadata& WithRuleOrder(int value) {
    SetRuleOrder(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The name of the rule. </p>
   */
  inline const Aws::String& GetRuleName() const { return m_ruleName; }
  inline bool RuleNameHasBeenSet() const { return m_ruleNameHasBeenSet; }
  template <typename RuleNameT = Aws::String>
  void SetRuleName(RuleNameT&& value) {
    m_ruleNameHasBeenSet = true;
    m_ruleName = std::forward<RuleNameT>(value);
  }
  template <typename RuleNameT = Aws::String>
  AutomationRulesMetadata& WithRuleName(RuleNameT&& value) {
    SetRuleName(std::forward<RuleNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A description of the rule. </p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  AutomationRulesMetadata& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether a rule is the last to be applied with respect to a finding
   * that matches the rule criteria. This is useful when a finding matches the
   * criteria for multiple rules, and each rule has different actions. If a rule is
   * terminal, Security Hub applies the rule action to a finding that matches the
   * rule criteria and doesn't evaluate other rules for the finding. By default, a
   * rule isn't terminal. </p>
   */
  inline bool GetIsTerminal() const { return m_isTerminal; }
  inline bool IsTerminalHasBeenSet() const { return m_isTerminalHasBeenSet; }
  inline void SetIsTerminal(bool value) {
    m_isTerminalHasBeenSet = true;
    m_isTerminal = value;
  }
  inline AutomationRulesMetadata& WithIsTerminal(bool value) {
    SetIsTerminal(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A timestamp that indicates when the rule was created. </p> <p>For more
   * information about the validation and formatting of timestamp fields in Security
   * Hub, see <a
   * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/Welcome.html#timestamps">Timestamps</a>.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedAt() const { return m_createdAt; }
  inline bool CreatedAtHasBeenSet() const { return m_createdAtHasBeenSet; }
  template <typename CreatedAtT = Aws::Utils::DateTime>
  void SetCreatedAt(CreatedAtT&& value) {
    m_createdAtHasBeenSet = true;
    m_createdAt = std::forward<CreatedAtT>(value);
  }
  template <typename CreatedAtT = Aws::Utils::DateTime>
  AutomationRulesMetadata& WithCreatedAt(CreatedAtT&& value) {
    SetCreatedAt(std::forward<CreatedAtT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A timestamp that indicates when the rule was most recently updated. </p>
   * <p>For more information about the validation and formatting of timestamp fields
   * in Security Hub, see <a
   * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/Welcome.html#timestamps">Timestamps</a>.</p>
   */
  inline const Aws::Utils::DateTime& GetUpdatedAt() const { return m_updatedAt; }
  inline bool UpdatedAtHasBeenSet() const { return m_updatedAtHasBeenSet; }
  template <typename UpdatedAtT = Aws::Utils::DateTime>
  void SetUpdatedAt(UpdatedAtT&& value) {
    m_updatedAtHasBeenSet = true;
    m_updatedAt = std::forward<UpdatedAtT>(value);
  }
  template <typename UpdatedAtT = Aws::Utils::DateTime>
  AutomationRulesMetadata& WithUpdatedAt(UpdatedAtT&& value) {
    SetUpdatedAt(std::forward<UpdatedAtT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The principal that created a rule. </p>
   */
  inline const Aws::String& GetCreatedBy() const { return m_createdBy; }
  inline bool CreatedByHasBeenSet() const { return m_createdByHasBeenSet; }
  template <typename CreatedByT = Aws::String>
  void SetCreatedBy(CreatedByT&& value) {
    m_createdByHasBeenSet = true;
    m_createdBy = std::forward<CreatedByT>(value);
  }
  template <typename CreatedByT = Aws::String>
  AutomationRulesMetadata& WithCreatedBy(CreatedByT&& value) {
    SetCreatedBy(std::forward<CreatedByT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_ruleArn;

  RuleStatus m_ruleStatus{RuleStatus::NOT_SET};

  int m_ruleOrder{0};

  Aws::String m_ruleName;

  Aws::String m_description;

  bool m_isTerminal{false};

  Aws::Utils::DateTime m_createdAt{};

  Aws::Utils::DateTime m_updatedAt{};

  Aws::String m_createdBy;
  bool m_ruleArnHasBeenSet = false;
  bool m_ruleStatusHasBeenSet = false;
  bool m_ruleOrderHasBeenSet = false;
  bool m_ruleNameHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_isTerminalHasBeenSet = false;
  bool m_createdAtHasBeenSet = false;
  bool m_updatedAtHasBeenSet = false;
  bool m_createdByHasBeenSet = false;
};

}  // namespace Model
}  // namespace SecurityHub
}  // namespace Aws
